#include "auth.h"

#include "exception.h"

#include <passport/infra/libs/cpp/juggler/status.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <library/cpp/tvmauth/client/facade.h>

#include <algorithm>

namespace NPassport::NKolmogor {
    const TString TAuth::SELF_ALIAS = "kolmogor";

    TAuth::TAuth(std::unique_ptr<NTvmAuth::TTvmClient> tvmClient, const TAuthSettings& cfg)
        : SelfClientId_({cfg.SelfTvmId})
        , TvmClient_(std::move(tvmClient))
    {
        TLog::Info() << "Authorization successfully enabled";
    }

    TAuth::~TAuth() = default;

    NJuggler::TStatus TAuth::GetStatus() const {
        NJuggler::TStatus status;
        status.Update(TvmClient_->GetStatus());
        return status;
    }

    void TAuth::AddAcl(TString space, const TAuth::TAllowedServices& clients) {
        if (clients.empty()) {
            return;
        }

        TAuth::TAllowedServices& c = Acl_.emplace(std::move(space), clients).first->second;
        std::sort(c.begin(), c.end());
    }

    std::optional<ui32> TAuth::GetSrcFromServiceTicket(TStringBuf ticket) const {
        if (!ticket) {
            return {};
        }

        const NTvmAuth::TCheckedServiceTicket t = TvmClient_->CheckServiceTicket(ticket);
        if (!t) {
            throw TAuthException()
                << "Invalid service ticket. " << NTvmAuth::StatusToString(t.GetStatus())
                << ": " << NTvmAuth::NUtils::RemoveTicketSignature(ticket);
        }

        return t.GetSrc();
    }

    void TAuth::CheckServiceTicketImpl(std::optional<ui32> tvmid, const TAuth::TAllowedServices& s, TStringBuf msg) {
        if (!tvmid) {
            throw TAuthException()
                << "Service ticket required but not found for '" << msg << "'";
        }

        if (!std::binary_search(s.begin(), s.end(), *tvmid)) {
            throw TAuthException()
                << "Service ticket is not allowed for '" << msg << "': " << *tvmid;
        }
    }

    void TAuth::CheckServiceTicketForReplication(TStringBuf ticket) const {
        CheckServiceTicketImpl(GetSrcFromServiceTicket(ticket), SelfClientId_, "replication");
    }

    TString TAuth::GetServiceTicketForReplication() const {
        return TvmClient_->GetServiceTicketFor(SELF_ALIAS);
    }

    void TAuth::CheckServiceTicket(std::optional<ui32> tvmid, const TSmallVec<TStringBuf>& spaces) const {
        for (const TStringBuf space : spaces) {
            auto it = Acl_.find(space);
            if (it == Acl_.end()) {
                continue;
            }

            if (!tvmid) {
                throw TAuthException()
                    << "Service ticket required but not found for '" << space << "'";
            }

            if (!std::binary_search(it->second.begin(), it->second.end(), *tvmid)) {
                throw TAuthException()
                    << "Service ticket is not allowed for '" << space << "'. Your tvm_id: " << *tvmid;
            }
        }
    }

    void TAuth::CheckRequiredServiceTicket(std::optional<ui32> tvmid, const TString& space) const {
        auto it = Acl_.find(space);
        if (it == Acl_.end()) {
            throw TAuthException()
                << "Auth was not configured for space '" << space << "'";
        }

        CheckServiceTicketImpl(tvmid, it->second, space);
    }
}
