#include "get_request.h"

#include "common.h"

#include <passport/infra/daemons/kolmogor/src/common/exception.h>

#include <passport/infra/libs/cpp/json/reader.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>

namespace NPassport::NKolmogor::NV2 {
    static TGetRequest::TSpace ParseSpace(rapidjson::Value::MemberIterator it) {
        TGetRequest::TSpace res;
        res.Name = TString(it->name.GetString(), it->name.GetStringLength());

        const rapidjson::Value* jsonKeys = nullptr;
        if (!NJson::TReader::MemberAsArray(it->value, "keys", jsonKeys)) {
            throw TBadRequestException() << "In space '" << res.Name << "' must be array 'keys'";
        }

        if (!jsonKeys->IsArray()) {
            throw TBadRequestException() << "Value must be array in '/" << res.Name << "/keys'";
        }

        res.Keys.reserve(jsonKeys->Size());
        for (size_t idx = 0; idx < jsonKeys->Size(); ++idx) {
            const rapidjson::Value& key = (*jsonKeys)[idx];
            if (!key.IsString()) {
                throw TBadRequestException()
                    << "Expecting array of strings in keys for '" << res.Name << "'"
                    << ". Got " << Type(key.GetType());
            }

            res.Keys.push_back(TString(key.GetString(), key.GetStringLength()));
        }

        return res;
    }

    TGetRequest TGetParser::Parse(const TString& body) {
        rapidjson::Document doc;
        if (!NJson::TReader::DocumentAsObject(body, doc)) {
            TLog::Debug() << "Bad json (base64): " << NUtils::BinToBase64(body);
            throw TBadRequestException() << "Invalid json";
        }

        TGetRequest req;
        req.Req.reserve(doc.MemberCount());

        for (auto it = doc.MemberBegin(); it != doc.MemberEnd(); ++it) {
            TGetRequest::TSpace reqSpace = ParseSpace(it);

            if (reqSpace.Keys.empty()) {
                throw TBadRequestException() << "There is no one key for space '" << reqSpace.Name << "'";
            }

            req.Req.push_back(std::move(reqSpace));
        }

        if (req.Req.empty()) {
            throw TBadRequestException() << "There is no one space in request";
        }

        return req;
    }
}
