#include "inc_request.h"

#include "common.h"

#include <passport/infra/daemons/kolmogor/src/common/exception.h>

#include <passport/infra/libs/cpp/json/reader.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <set>

namespace NPassport::NKolmogor::NV2 {
    static void ThrowIncIfLess(const TStringBuf k) {
        throw TBadRequestException()
            << "key '" << k << "' was used in several keysets and with 'inc_if_less_than';"
            << " this scenerio is not supported";
    }

    static size_t ParseKeyset(rapidjson::Value& keySet,
                              const TString& name,
                              TIncRequest::TKeySet& keyset,
                              std::set<TStringBuf>& usualKeys,
                              std::set<TStringBuf>& conditionalKeys) {
        if (!keySet.IsObject()) {
            throw TBadRequestException() << "Array for space '" << name << "' must contain only objects";
        }

        const rapidjson::Value* jsonKeys = nullptr;
        if (!NJson::TReader::MemberAsArray(keySet, "keys", jsonKeys)) {
            throw TBadRequestException() << "In every object for space '" << name << "' must be array 'keys'";
        }

        keyset.Keys.reserve(jsonKeys->Size());

        for (size_t idx = 0; idx < jsonKeys->Size(); ++idx) {
            const rapidjson::Value& key = (*jsonKeys)[idx];
            if (!key.IsString()) {
                throw TBadRequestException()
                    << "Expecting array of strings '" << name << "'"
                    << ". Got " << Type(key.GetType());
            }

            keyset.Keys.push_back(TString(key.GetString(), key.GetStringLength()));
        }

        if (auto it = keySet.FindMember("inc_if_less_than"); it != keySet.MemberEnd()) {
            if (!it->value.IsUint64()) {
                throw TBadRequestException()
                    << "'inc_if_less_than' must be uint64. Got " << Type(it->value.GetType());
            }
            keyset.IncIfLessThan = it->value.GetUint64();

            for (const TStringBuf k : keyset.Keys) {
                if (conditionalKeys.contains(k)) {
                    ThrowIncIfLess(k);
                }
            }
            // just to allow doubles in the same keyset we need to insert after all checks
            conditionalKeys.insert(keyset.Keys.begin(), keyset.Keys.end());
        } else {
            usualKeys.insert(keyset.Keys.begin(), keyset.Keys.end());
        }

        return jsonKeys->Size();
    }

    static TIncRequest::TSpace ParseSpace(rapidjson::Value::MemberIterator it) {
        TIncRequest::TSpace res;
        res.Name = TString(it->name.GetString(), it->name.GetStringLength());

        if (!it->value.IsArray()) {
            throw TBadRequestException() << "Value must be array for space '" << res.Name << "'";
        }

        res.Keysets.reserve(it->value.Size());

        std::set<TStringBuf> usualKeys;
        std::set<TStringBuf> conditionalKeys;

        for (size_t idx = 0; idx < it->value.Size(); ++idx) {
            res.Keysets.push_back({});
            res.KeysCount += ParseKeyset(
                it->value[idx],
                res.Name,
                res.Keysets.back(),
                usualKeys,
                conditionalKeys);
        }

        std::vector<TStringBuf> intersection;
        std::set_intersection(usualKeys.begin(), usualKeys.end(),
                              conditionalKeys.begin(), conditionalKeys.end(),
                              std::back_inserter(intersection));
        if (!intersection.empty()) {
            ThrowIncIfLess(intersection[0]);
        }

        return res;
    }

    TIncRequest TIncParser::Parse(const TString& body) {
        rapidjson::Document doc;
        if (!NJson::TReader::DocumentAsObject(body, doc)) {
            TLog::Debug() << "Bad json (base64): " << NUtils::BinToBase64(body);
            throw TBadRequestException() << "Invalid json";
        }

        TIncRequest req;
        req.Req.reserve(doc.MemberCount());

        for (auto it = doc.MemberBegin(); it != doc.MemberEnd(); ++it) {
            TIncRequest::TSpace reqSpace = ParseSpace(it);

            if (reqSpace.Keysets.empty()) {
                throw TBadRequestException() << "There is no one key for space '" << reqSpace.Name << "'";
            }

            req.Req.push_back(std::move(reqSpace));
        }

        if (req.Req.empty()) {
            throw TBadRequestException() << "There is no one space in request";
        }

        return req;
    }
}
