#include "kolmogor.h"

#include "common/auth.h"
#include "common/exception.h"
#include "common/utils.h"
#include "input/eraseall_request.h"
#include "output/get_result.h"
#include "storage/mem_storage.h"

#include <passport/infra/libs/cpp/json/config.h>
#include <passport/infra/libs/cpp/juggler/status.h>
#include <passport/infra/libs/cpp/request/request.h>
#include <passport/infra/libs/cpp/tvm/logger/logger.h>
#include <passport/infra/libs/cpp/utils/ipaddr.h>
#include <passport/infra/libs/cpp/utils/log/file_logger.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/format.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <library/cpp/json/writer/json.h>
#include <library/cpp/tvmauth/client/facade.h>

#include <util/system/hostname.h>

#include <regex>

namespace NPassport::NKolmogor {
    static const TString SPACE_ = "space";
    static const TString KEYS_ = "keys";
    static const TString CONTENT_TYPE_JSON = "application/json";
    static const TString CONTENT_TYPE_TEXT = "text/plain";
    static const TString CONTENT_TYPE = "Content-type";

    static const TString& GetRequieredArg(const NCommon::TRequest& req, const TString& arg) {
        const TString& s = req.GetArg(arg);
        if (!s) {
            throw TBadRequestException() << "arg is empty: '" << arg << "'";
        }
        return s;
    }

    TKolmogor::TKolmogor() = default;

    TKolmogor::~TKolmogor() {
        try {
            SaveOnExit();
        } catch (...) {
        }
    }

    void TKolmogor::Init(const NJson::TConfig& cfg) {
        const TString path = "/component";

        InitLoggers(cfg, path);
        InitAuth(cfg, path);
        InitStorage(cfg, path);
        InitReplicator(cfg, path);
        InitSpaces(cfg, path);
        InitMisc(cfg, path);

        LoadOnStart();
    }

    void TKolmogor::HandleRequest(NCommon::TRequest& req) {
        const TInstant start = TInstant::Now();

        std::optional<ui32> tvmid;
        TResult res = ProcessRequest(req, tvmid);

        req.SetStatus(res.Status);
        req.SetHeader(CONTENT_TYPE, res.ContentType);
        req.Write(res.Body);

        LogAccess(req, std::move(res), tvmid, TInstant::Now() - start);
    }

    void TKolmogor::AddUnistat(NUnistat::TBuilder& builder) {
        Storage_->AddUnistat(builder);
        Replicator_->AddUnistat(builder);
    }

    void TKolmogor::LoadOnStart() {
        Storage_->LoadOnStart();
        Replicator_->StartWork();
    }

    void TKolmogor::SaveOnExit() {
        if (Storage_) {
            Storage_->StopThreads();
        }

        TLog::Info() << "Stoping replication";
        if (Replicator_) {
            Replicator_->StopThreads();
        }
        Replicator_.reset();
        TLog::Info() << "Relication is stoped";

        if (Storage_) {
            Storage_->SaveOnExit();
        }
    }

    static const TString OKEY = "OK\n";
    static const TString BAD_AUTH = "Authorization failed\n";
    TResult TKolmogor::ProcessRequest(NCommon::TRequest& req, std::optional<ui32>& tvmid) {
        req.ForceProvideRequestId();

        const TString& path = req.GetPath();
        if (path.StartsWith("/2/")) {
            return ProcessV2(req, tvmid);
        }

        req.ScanCgiFromBody();
        return ProcessV1(req, path, tvmid);
    }

    void TKolmogor::CheckDuplicatedArgs(const NCommon::TRequest& req) {
        NCommon::TRequest::TDuplicatedArgs dups = req.GetDuplicatedArgs();
        if (!dups.empty()) {
            throw TBadRequestException()
                << "Duplicated args are not supporeted, got several values of '"
                << dups.begin()->first << "'";
        }
    }

    TResult TKolmogor::ProcessV1(const NCommon::TRequest& req, TStringBuf path, std::optional<ui32>& tvmid) {
        try {
            CheckDuplicatedArgs(req);
            GetSrcFromServiceTicket(req, tvmid);

            return ProcessV1Impl(req, path, tvmid);
        } catch (const TAuthException& e) {
            TLog::Debug() << "Service: Authorization failed: " << e.what();
            const TString msg = NUtils::CreateStr("Authorization failed: ", e.what(), '\n');
            return MakeResult(msg, HttpCodes::HTTP_FORBIDDEN); // 403
        } catch (const TBadRequestException& e) {
            TLog::Debug() << "Service: bad request: " << e.what();
            const TString msg = NUtils::CreateStr("Error: ", e.what(), '\n');
            return MakeResult(msg, HttpCodes::HTTP_BAD_REQUEST); // 400
        } catch (const TNotImplementedException& e) {
            TLog::Debug() << "Service: not implemented: " << e.what();
            const TString msg = NUtils::CreateStr("Not implemented: ", e.what(), '\n');
            return MakeResult(msg, HttpCodes::HTTP_NOT_IMPLEMENTED); // 501
        } catch (const std::exception& e) {
            TLog::Warning() << "Service: unexpected exception: " << e.what();
            const TString msg = NUtils::CreateStr("Error: ", e.what(), '\n');
            return MakeResult(msg, HttpCodes::HTTP_INTERNAL_SERVER_ERROR); // 500
        } catch (...) {
            TLog::Error() << "Service: fatal exception";
            return MakeResult("Error: fatal exception\n", HttpCodes::HTTP_INTERNAL_SERVER_ERROR); // 500
        }
    }

    TResult TKolmogor::ProcessV1Impl(const NCommon::TRequest& req, TStringBuf path, std::optional<ui32> tvmid) {
        const TString& method = req.GetRequestMethod();
        if (method != "GET" && method != "POST") {
            TString s = "POST and GET only allowed: ";
            TLog::Debug() << "POST and GET only allowed: " << method;
            return MakeResult(s + method + ".\n", HttpCodes::HTTP_BAD_REQUEST);
        }

        if (path == "/ping") {
            return HandlePing(req);
        }

        if (path == "/inc") {
            CheckAuth(req, tvmid);
            NV2::TIncResult v2 = HandleV1Inc(req);

            TResult res = MakeResult(OKEY);
            res.AccessLogString = NV2::TSerializer::Serialize(v2);
            return res;
        }

        if (path == "/get") {
            CheckAuth(req, tvmid);
            return HandleV1Get(req);
        }

        if (path == "/erase") {
            TResult res = HandleV1Erase(req, tvmid);
            res.AccessLogString = SerializeRequest(req);
            return res;
        }

        throw TBadRequestException() << "Path is unknown: " << path;
    }

    TResult TKolmogor::ProcessV2(const NCommon::TRequest& req, std::optional<ui32>& tvmid) {
        try {
            CheckDuplicatedArgs(req);
            GetSrcFromServiceTicket(req, tvmid);

            TResult res = ProcessV2Impl(req, tvmid);
            res.AccessLogString = res.Body;
            return res;
        } catch (const TAuthException& e) {
            TLog::Debug() << "Service: Authorization failed: " << e.what();
            return MakeResultV2(NV2::TError{NUtils::CreateStr("Authorization failed: ", e.what())},
                                HttpCodes::HTTP_FORBIDDEN);
        } catch (const TBadRequestException& e) {
            TLog::Debug() << "Service: bad request: " << e.what();
            return MakeResultV2(NV2::TError{NUtils::CreateStr("Bad request: ", e.what())},
                                HttpCodes::HTTP_BAD_REQUEST);
        } catch (const TNotImplementedException& e) {
            TLog::Debug() << "Service: not implemented: " << e.what();
            return MakeResultV2(NV2::TError{NUtils::CreateStr("Not implemented: ", e.what())},
                                HttpCodes::HTTP_NOT_IMPLEMENTED);
        } catch (const std::exception& e) {
            TLog::Warning() << "Service: unexpected exception: " << e.what();
            return MakeResultV2(NV2::TError{NUtils::CreateStr("Unexpected exception: ", e.what())},
                                HttpCodes::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    TResult TKolmogor::ProcessV2Impl(const NCommon::TRequest& req, std::optional<ui32> tvmid) {
        const TString& method = req.GetRequestMethod();
        if (method != "POST") {
            throw TBadRequestException() << "POST only allowed: " << method;
        }

        const TString& contentType = req.GetHeader(CONTENT_TYPE);
        if (contentType != CONTENT_TYPE_JSON) {
            throw TBadRequestException() << "Content-type must be '" << CONTENT_TYPE_JSON
                                         << "'. Got '" << contentType << "'";
        }

        const TString& path = req.GetPath();
        if (path == "/2/inc") {
            return MakeResultV2(HandleV2Inc(req.GetRequestBody(), tvmid));
        }

        if (path == "/2/get") {
            return MakeResultV2(HandleV2Get(req.GetRequestBody(), tvmid));
        }

        if (path == "/2/eraseall") {
            if (!NUtils::TIpAddr(req.GetRemoteAddr()).IsLoopback()) {
                throw TAuthException() << "Only loopback allowed for this request";
            }

            return MakeResultV2(HandleV2EraseAll(req.GetRequestBody()));
        }

        if (path == "/2/erase") {
            throw TNotImplementedException() << "Path: " << path;
        }

        throw TBadRequestException() << "Path is unknown: " << path;
    }

    static const TString DOWN = "Externally forced out-of-service\n";

    TResult TKolmogor::HandlePing(const NCommon::TRequest& req) const {
        const NJuggler::TStatus authStatus = Auth_->GetStatus();

        if (req.HasArg("check_alerts")) {
            return MakeResult(authStatus);
        }

        if (authStatus == NJuggler::ECode::Critical) {
            return MakeResult(authStatus.Message(), HttpCodes::HTTP_INTERNAL_SERVER_ERROR); // 500
        }

        if (TUtils::IsForceDown(ForceDownFile_)) {
            TLog::Info() << "Ping handler: Externally forced out-of-service";
            return MakeResult(DOWN, HttpCodes::HTTP_SERVICE_UNAVAILABLE); // 503
        }

        return MakeResult(OKEY);
    }

    NV2::TIncResult TKolmogor::HandleV1Inc(const NCommon::TRequest& req) {
        const TString& space = GetRequieredArg(req, SPACE_);
        const TString& keys = GetRequieredArg(req, KEYS_);

        NV2::TIncRequest::TSpace sp;
        sp.Name = space;
        sp.Keysets.emplace_back();
        TUtils::Split(keys, ',', sp.Keysets.back().Keys);

        NV2::TIncRequest incReq;
        incReq.Req.push_back(std::move(sp));

        return Storage_->IncFromHttp(std::move(incReq));
    }

    TResult TKolmogor::HandleV1Get(const NCommon::TRequest& req) const {
        const TString& space = GetRequieredArg(req, SPACE_);
        const TString& keys = GetRequieredArg(req, KEYS_);

        NV2::TGetRequest::TSpace sp;
        sp.Name = space;
        TUtils::Split(keys, ',', sp.Keys);

        NV2::TGetRequest getReq;
        getReq.Req.push_back(std::move(sp));

        NV2::TGetResult res = Storage_->Get(getReq, TSpace::EAPI::V1);

        TStringStream s;
        s.Reserve(5 * res.Data.at(0).Values.size());
        for (auto n : res.Data.at(0).Values) {
            s << n.Value << ',';
        }
        // Replace last comma by newliner
        if (s.Str()) {
            s.Str().back() = '\n';
        }
        s.Str();

        TResult r = MakeResult(s.Str());
        r.AccessLogString = NV2::TSerializer::Serialize(res);
        return r;
    }

    TResult TKolmogor::HandleV1Erase(const NCommon::TRequest& req, std::optional<ui32> tvmid) {
        const TString& space = GetRequieredArg(req, SPACE_);
        const TString& keys = GetRequieredArg(req, KEYS_);

        if (!Storage_->HasSpace(space)) {
            throw TBadRequestException() << "Space is not found (erase): " << space;
        }
        CheckRequiredAuth(req, tvmid);

        TString res = Storage_->Erase(space, keys);
        return res ? MakeResult(res, HTTP_PARTIAL_CONTENT) : MakeResult(OKEY, HTTP_OK);
    }

    NV2::TIncResult TKolmogor::HandleV2Inc(const TString& body, std::optional<ui32> tvmid) {
        NV2::TIncRequest req = NV2::TIncParser::Parse(body);
        CheckAuth(req, tvmid);
        return Storage_->IncFromHttp(std::move(req));
    }

    NV2::TGetResult TKolmogor::HandleV2Get(const TString& body, std::optional<ui32> tvmid) const {
        NV2::TGetRequest req = NV2::TGetParser::Parse(body);
        CheckAuth(req, tvmid);
        return Storage_->Get(req, TSpace::EAPI::V2);
    }

    NV2::TEraseAllResult TKolmogor::HandleV2EraseAll(const TString& body) {
        NV2::TEraseAllRequest req = NV2::TEraseAllParser::Parse(body);
        return Storage_->EraseAll(req);
    }

    TResult TKolmogor::MakeResult(const TString& str, HttpCodes code) {
        return MakeResult(str, CONTENT_TYPE_TEXT, code);
    }

    TResult TKolmogor::MakeResult(const TString& str, const TString& contType, HttpCodes code) {
        return TResult{
            .Status = code,
            .Body = str,
            .ContentType = contType,
        };
    }

    template <class T>
    TResult TKolmogor::MakeResultV2(const T& result, HttpCodes code) const {
        return TResult{
            .Status = code,
            .Body = NV2::TSerializer::Serialize(result),
            .ContentType = CONTENT_TYPE_JSON,
        };
    }

    void TKolmogor::CheckAuth(const NCommon::TRequest& req, std::optional<ui32> tvmid) const {
        Auth_->CheckServiceTicket(
            tvmid,
            {GetRequieredArg(req, SPACE_)});
    }

    template <class T>
    void TKolmogor::CheckAuth(const T& req, std::optional<ui32> tvmid) const {
        TSmallVec<TStringBuf> spaces;
        spaces.reserve(req.Req.size());

        for (const typename T::TSpace& s : req.Req) {
            spaces.push_back(s.Name);
        }

        Auth_->CheckServiceTicket(tvmid, spaces);
    }

    void TKolmogor::CheckRequiredAuth(const NCommon::TRequest& req, std::optional<ui32> tvmid) const {
        Auth_->CheckRequiredServiceTicket(
            tvmid,
            GetRequieredArg(req, SPACE_));
    }

    static const TString X_YA_SERVICE_TICKET = "X-Ya-Service-Ticket";

    void TKolmogor::GetSrcFromServiceTicket(const NCommon::TRequest& req, std::optional<ui32>& out) const {
        out = Auth_->GetSrcFromServiceTicket(req.GetHeader(X_YA_SERVICE_TICKET));
    }

    TString TKolmogor::SerializeRequest(const NCommon::TRequest& req) {
        TString res = req.GetQueryString();
        NUtils::AppendSeparated(res, '&', req.GetRequestBody());

        return res;
    }

    static const TString EXTERNAL_REQ_ID = "X-Ext-Request-Id";
    static const TString DASH = "-";
    static const TString DELIM = "\t";

    void TKolmogor::LogAccess(const NCommon::TRequest& req,
                              TResult&& res,
                              std::optional<ui32> tvmid,
                              TDuration time) const {
        if (!AccessLog_ ||
            req.GetPath() == "/ping" ||
            req.GetPath() == "/stat") {
            return;
        }

        const TString& extReqId = req.GetHeader(EXTERNAL_REQ_ID);

        char buf[16];
        TStringBuf tvmidBuf = DASH;
        if (tvmid) {
            tvmidBuf = TStringBuf(buf, IntToString<10>(*tvmid, buf, sizeof(buf)));
        }

        NUtils::EscapeEol(res.AccessLogString);

        // 'ERROR' level allowes to skip record 'file was reopened' on rotation
        AccessLog_->Error(1024)
            << (extReqId ? extReqId : DASH) << DELIM
            << req.GetRemoteAddr() << DELIM
            << time.MicroSeconds() << "us" << DELIM
            << (int)req.GetStatusCode() << DELIM
            << tvmidBuf << DELIM
            << req.GetRequestMethod() << DELIM
            << req.GetPath() << DELIM
            << (res.AccessLogString ? res.AccessLogString : DASH);
    }

    void TKolmogor::InitLoggers(const NJson::TConfig& config, const TString& path) {
        config.InitCommonLog(path + "/logger");

        if (config.Contains(path + "/access_log")) {
            AccessLog_ = config.CreateLogger(path + "/access_log");
        }
    }

    void TKolmogor::InitAuth(const NJson::TConfig& config, const TString& path) {
        NJson::TConfig tvm = NJson::TConfig::ReadFromFile(config.As<TString>(path + "/tvm/config"));

        TAuthSettings auth{
            .SelfTvmId = tvm.As<ui32>("/tvm_id"),
        };

        NTvmAuth::NTvmApi::TClientSettings settings;
        settings.SetDiskCacheDir(config.As<TString>(path + "/tvm/cache"));
        settings.SetSelfTvmId(auth.SelfTvmId);
        settings.EnableServiceTicketChecking();
        settings.EnableServiceTicketsFetchOptions(
            tvm.As<TString>("/tvm_secret"),
            {{TAuth::SELF_ALIAS, auth.SelfTvmId}});

        if (config.Contains(path + "/tvm/local_port")) {
            settings.SetTvmHostPort("localhost", config.As<ui32>(path + "/tvm/local_port"));
        }

        Auth_ = std::make_unique<TAuth>(
            std::make_unique<NTvmAuth::TTvmClient>(
                settings,
                NTvmLogger::TLogger::Create()),
            auth);
    }

    void TKolmogor::InitStorage(const NJson::TConfig& config, const TString& path) {
        TMemStorageSettings settings{
            .DataDirectory = config.As<TString>(path + "/storage/data_directory", "/var/lib/kolmogor/data"),
            .MinCleanPeriod = TDuration::Seconds(config.As<ui32>(path + "/storage/min_clean_period_sec", 30)),
            .CleanCount = config.As<ui32>(path + "/storage/clean_count_per_shot", 512),
            .ThreadsForDump = config.As<ui32>(path + "/storage/threads_for_dump", 64),
        };
        Storage_ = std::make_unique<TMemStorage>(settings);
    }

    void TKolmogor::InitReplicator(const NJson::TConfig& config, const TString& path) {
        TReplicatorSettings settings{
            .Port = (ui16)config.As<ui32>(path + "/replication/port"),
            .Threads = config.As<ui32>(path + "/replication/threads", 8),
            .PingPeriod = TDuration::Seconds(config.As<ui32>(path + "/replication/ping_period_sec", 3)),
            .Timeout = TDuration::Seconds(config.As<ui32>(path + "/replication/timeout_sec", 1)),
            .GroupCount = config.As<ui32>(path + "/replication/group_count", 128),
            .DebtCleanPeriod = TDuration::Seconds(config.As<ui32>(path + "/replication/debt_clean_period_sec", 300)),
            .EraseRetries = config.As<ui32>(path + "/replication/erase_retries", 3),
            .EraseTimeout = TDuration::MilliSeconds(config.As<ui32>(path + "/replication/erase_timeout_ms", 300)),
            .MaxMessageSize = config.As<int>(path + "/replication/max_message_size", 32 * 1024 * 1024),
            .MaxDebtSize = config.As<ui64>(path + "/replication/max_queue_size", 0),
        };

        const TString localhost = FQDNHostName();
        const std::regex rgx(R"([\w\d\.-]+:[\d]+)");

        for (const TString& h : config.SubKeys(path + "/replication/dest")) {
            TString cur = config.As<TString>(h);
            Y_ENSURE_EX(std::regex_match(cur.cbegin(), cur.cend(), rgx),
                        TConfigException() << "Uri '" << h << "' is invalid");

            TStringBuf hport = cur;
            TStringBuf hname = hport.NextTok(':');
            if (localhost == hname ||
                (hname == "localhost" && settings.Port == IntFromString<ui16, 10>(hport))) {
                continue;
            }

            settings.Uris.push_back(cur);
        }

        Replicator_ = std::make_shared<TReplicator>(*Storage_, *Auth_, settings);
        Storage_->SetReplicator(Replicator_);
    }

    void TKolmogor::InitSpaces(const NJson::TConfig& config, const TString& path) {
        for (const TString& s : config.SubKeys(path + "/spaces")) {
            TSpaceSettings settings{
                .Name = config.As<TString>(s + "/name"),
                .SliceCount = config.As<size_t>(s + "/slice_count", 2048),
                .Reserve = config.As<size_t>(s + "/slice_reserve", 2097152),
                .CounterTtl = config.As<ui32>(s + "/num_ttl"),
                .CounterCount = config.As<ui32>(s + "/splited_count"),
                .ThreadCount = config.As<ui32>(s + "/thread_count", 1),
                .EraseCount = config.As<size_t>(s + "/erase_count", 1024),
                .Persistency = config.As<bool>(s + "/persistency", true),
            };
            Y_ENSURE_EX(settings.CounterTtl >= settings.CounterCount,
                        TConfigException() << "Invalid config: " << s << ": /num_ttl less then /splited_count");

            // '/' is used in unistat signals
            Y_ENSURE(!settings.Name.Contains("/"),
                     "space name cannot contain '/': '" << settings.Name << "'");

            if (config.Contains(s + "/allowed_client_id")) {
                for (const TString& a : config.SubKeys(s + "/allowed_client_id")) {
                    settings.AllowedClientIds.push_back(config.As<ui32>(a));
                }
            }
            if (config.Contains(s + "/memory_limit")) {
                settings.MemoryLimit = config.As<size_t>(s + "/memory_limit");
            }

            Storage_->AddSpace(settings);
            Replicator_->AddSpace(settings.Name, settings.ThreadCount);
            Auth_->AddAcl(settings.Name, settings.AllowedClientIds);
        }

        Storage_->StartCleaning();
    }

    void TKolmogor::InitMisc(const NJson::TConfig& config, const TString& path) {
        ForceDownFile_ = config.As<TString>(path + "/misc/force_down_file", "/var/run/kolmogor.down");
    }
}
