#include "client.h"

#include "request/ping_request.h"

#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/log/global.h>

#include <contrib/libs/grpc/include/grpcpp/create_channel.h>

namespace NPassport::NKolmogor {
    static auto CreateChanel(const TString& uri, int maxMessageSize) {
        grpc::ChannelArguments args;
        args.SetMaxReceiveMessageSize(maxMessageSize);
        args.SetMaxSendMessageSize(maxMessageSize);
        return grpc::CreateCustomChannel(uri, grpc::InsecureChannelCredentials(), args);
    }

    static const std::map<grpc_connectivity_state, TString> GRPC_CHANNEL_STATUS_TO_STRING = {
        {GRPC_CHANNEL_IDLE, "GRPC_CHANNEL_IDLE"},
        {GRPC_CHANNEL_CONNECTING, "GRPC_CHANNEL_CONNECTING"},
        {GRPC_CHANNEL_READY, "GRPC_CHANNEL_READY"},
        {GRPC_CHANNEL_TRANSIENT_FAILURE, "GRPC_CHANNEL_TRANSIENT_FAILURE"},
        {GRPC_CHANNEL_SHUTDOWN, "GRPC_CHANNEL_SHUTDOWN"},
    };

    static TStringBuf ChannelStatus(grpc_connectivity_state s) {
        auto it = GRPC_CHANNEL_STATUS_TO_STRING.find(s);
        return it == GRPC_CHANNEL_STATUS_TO_STRING.end()
                   ? TStringBuf()
                   : it->second;
    }

    TClient::TClient(const TString& uri,
                     grpc::CompletionQueue& cq,
                     const TAuth& auth,
                     const TClientSettings& settings,
                     NUnistat::TSignalDiff<>& unistatAllErrors)
        : Uri_(uri)
        , HostInSignal_(NUtils::ReplaceAny(Uri_, ":", "_"))
        , IsDown_(true)
        , Auth_(auth)
        , ClientCq_(cq)
        , Channel_(CreateChanel(uri, settings.MaxMessageSize))
        , Stub_(kolmogor::replication::v2::Repl::NewStub(Channel_))
        , Timeout_(settings.Timeout)
        , DebtDispatcher_(settings.DebtDispatcher)
        , UnistatAllErrors_(unistatAllErrors)
    {
        auto add = [&](EClientErrors err, TString id) {
            UnistatErrors_.emplace(
                err,
                std::make_unique<NUnistat::TSignalDiff<>>(
                    "replication.host/" + HostInSignal_ + "/errors." + id));
        };

        add(EClientErrors::Timeout, "timeout");
        add(EClientErrors::NetworkError, "network");
        add(EClientErrors::Other, "other");

        UnistatDebtSize_ = std::make_unique<NUnistat::TSignalAbsolute<>>(
            "replication.host/" + HostInSignal_ + "/debt_requests", NUnistat::NSuffix::AXXX);
    }

    void TClient::AddUnistat(NUnistat::TBuilder& builder) {
        for (const auto& [err, sig] : UnistatErrors_) {
            builder.Add(*sig);
        }
        builder.Add(*UnistatDebtSize_);
    }

    void TClient::SendPing() {
        std::unique_ptr req = std::make_unique<TPingRequest>(*this, ClientCq_);

        req->Send(*Stub_, Auth_, Timeout_);
        Y_UNUSED(req.release()); // to be cleaned by Accepter
    }

    void TClient::SendPush(TTailPtr tail, time_t expireTime) {
        std::unique_ptr req = std::make_unique<TPushRequest>(*this, ClientCq_, std::move(tail));

        if (IsDown_.load(std::memory_order_relaxed)) {
            AddDebt(std::make_shared<TNode>(TNode{
                .Request = std::move(req),
                .ExpireTime = expireTime,
            }));
            return;
        }

        req->Send(*Stub_, Auth_, Timeout_);
        Y_UNUSED(req.release()); // to be cleaned by Accepter
    }

    std::unique_ptr<TEraseRequest> TClient::SendErase(TToErasePtr toErase,
                                                      grpc::CompletionQueue& cq,
                                                      TDuration timeout) {
        std::unique_ptr req = std::make_unique<TEraseRequest>(*this, cq, std::move(toErase));

        req->Send(*Stub_, Auth_, timeout);

        return req;
    }

    void TClient::SendErase(TToErasePtr toErase, time_t expireTime) {
        std::unique_ptr req = std::make_unique<TEraseRequest>(*this, ClientCq_, std::move(toErase));

        if (IsDown_.load(std::memory_order_relaxed)) {
            AddDebt(std::make_shared<TNode>(TNode{
                .Request = std::move(req),
                .ExpireTime = expireTime,
            }));
            return;
        }

        req->Send(*Stub_, Auth_, Timeout_);
        Y_UNUSED(req.release()); // to be cleaned by Accepter
    }

    void TClient::AddDebt(TBaseRequestPtr req, time_t expireTime) {
        const time_t now = time(nullptr);
        if (expireTime <= now) {
            TLog::Debug() << "Debt is expired for " << Uri_ << " when tried to add it";
            return;
        }

        AddDebt(std::make_shared<TNode>(TNode{
            .Request = std::move(req),
            .ExpireTime = expireTime,
        }));
    }

    bool TClient::SendSomeDebt() {
        if (IsDown_.load(std::memory_order_relaxed)) {
            return false;
        }

        const time_t now = time(nullptr);

        TNodePtr node;
        while ((node = PopDebt())) {
            if (node->ExpireTime <= now) {
                TLog::Debug() << "Debt is expired for " << Uri_ << " when tried to send it";
                continue;
            }

            node->Request->Send(*Stub_, Auth_, Timeout_);
            Y_UNUSED(node->Request.release()); // to be cleaned by Accepter

            TLog::Debug() << "Debt is sent for " << Uri_;
            return true;
        }

        return false;
    }

    void TClient::MakeOk() {
        IsDown_.store(false, std::memory_order_relaxed);
    }

    void TClient::MakeDown(EClientErrors err) {
        IsDown_.store(true, std::memory_order_relaxed);

        ++UnistatAllErrors_;
        auto it = UnistatErrors_.find(err);
        if (it != UnistatErrors_.end()) {
            ++(*it->second);
        }
    }

    void TClient::CleanDebt() {
        TNodePtr node;
        while ((node = PopDebt())) {
            if (node->ExpireTime > time(nullptr)) {
                AddDebt(std::move(node));
                return;
            }

            TLog::Debug() << "Debt is expired for " << Uri_;
        }
    }

    bool TClient::AddDebt(TNodePtr node) {
        ui64 requestSize = node->Request->GetRequestSize();
        if (DebtDispatcher_ && !DebtDispatcher_->TryAcquire(HostInSignal_, requestSize)) {
            TLog::Debug() << "Debt is not added for " << Uri_ << ". Debt Queue is full";
            return false;
        }

        Debt_.Enqueue(std::move(node));
        ++(*UnistatDebtSize_);
        TLog::Debug() << "Debt is added for " << Uri_
                      << ". channel: " << ChannelStatus(Channel_->GetState(false));
        return true;
    }

    TClient::TNodePtr TClient::PopDebt() {
        TNodePtr node;

        if (Debt_.Dequeue(&node)) {
            if (DebtDispatcher_) {
                ui64 requestSize = node->Request->GetRequestSize();
                Y_ENSURE(DebtDispatcher_->Release(HostInSignal_, requestSize));
            }
            --(*UnistatDebtSize_);
        }

        return node;
    }
}
