#pragma once

#include "request/erase_request.h"
#include "request/push_request.h"

#include <passport/infra/daemons/kolmogor/src/proto/replication.v2.grpc.pb.h>

#include <passport/infra/libs/cpp/logbroker/resource_dispatcher/resource_dispatcher.h>
#include <passport/infra/libs/cpp/unistat/absolute.h>
#include <passport/infra/libs/cpp/unistat/diff.h>

#include <contrib/libs/grpc/include/grpcpp/channel.h>
#include <contrib/libs/grpc/include/grpcpp/completion_queue.h>

#include <util/generic/string.h>
#include <util/thread/lfqueue.h>

#include <atomic>
#include <map>
#include <memory>

namespace NPassport::NUnistat {
    class TBuilder;
}

namespace NPassport::NKolmogor {
    enum class EClientErrors {
        Timeout,
        NetworkError,
        Other,
    };

    struct TClientSettings {
        TDuration Timeout;
        int MaxMessageSize = 0;
        std::shared_ptr<NLb::TResourceDispatcher> DebtDispatcher;
    };

    class TClient {
    public:
        TClient(const TString& uri,
                grpc::CompletionQueue& cq,
                const TAuth& auth,
                const TClientSettings& settings,
                NUnistat::TSignalDiff<>& unistatAllErrors);

        void AddUnistat(NUnistat::TBuilder& builder);

        void SendPing();
        void SendPush(TTailPtr tail, time_t expireTime);
        std::unique_ptr<TEraseRequest> SendErase(TToErasePtr toErase,
                                                 grpc::CompletionQueue& cq,
                                                 TDuration timeout);
        void SendErase(TToErasePtr toErase, time_t expireTime);

        void AddDebt(TBaseRequestPtr req, time_t expireTime);
        bool SendSomeDebt();
        void MakeOk();
        void MakeDown(EClientErrors err);

        void CleanDebt();

        const TString& Uri() const {
            return Uri_;
        }

        const TString& HostInSignal() const {
            return HostInSignal_;
        }

    private:
        struct TNode {
            TBaseRequestPtr Request;
            time_t ExpireTime;
        };
        using TNodePtr = std::shared_ptr<TNode>;

    private:
        bool AddDebt(TNodePtr node);
        TNodePtr PopDebt();

    private:
        const TString Uri_;
        // YASM doesn't support ':' in signal name
        const TString HostInSignal_;

        std::atomic_bool IsDown_;
        const TAuth& Auth_;
        grpc::CompletionQueue& ClientCq_;

        std::shared_ptr<grpc::Channel> Channel_;
        std::unique_ptr<kolmogor::replication::v2::Repl::Stub> Stub_;

        TDuration Timeout_;
        TLockFreeQueue<TNodePtr> Debt_;
        std::shared_ptr<NLb::TResourceDispatcher> DebtDispatcher_;
        std::map<EClientErrors, std::unique_ptr<NUnistat::TSignalDiff<>>> UnistatErrors_;
        NUnistat::TSignalDiff<>& UnistatAllErrors_;
        std::unique_ptr<NUnistat::TSignalAbsolute<>> UnistatDebtSize_;
    };

    using TClientPtr = std::shared_ptr<TClient>;
}

template <>
struct std::less<NPassport::NKolmogor::TClientPtr> {
    bool operator()(const NPassport::NKolmogor::TClientPtr& lhs, const NPassport::NKolmogor::TClientPtr& rhs) const {
        return lhs->Uri() < rhs->Uri();
    }
    bool operator()(const TString& lhs, const NPassport::NKolmogor::TClientPtr& rhs) const {
        return lhs < rhs->Uri();
    }
    bool operator()(const NPassport::NKolmogor::TClientPtr& lhs, const TString& rhs) const {
        return lhs->Uri() < rhs;
    }
    using is_transparent = std::true_type;
};
