#include "eraser.h"

#include <passport/infra/daemons/kolmogor/src/common/auth.h>

#include <passport/infra/libs/cpp/utils/log/global.h>

#include <util/string/builder.h>

namespace NPassport::NKolmogor {
    TEraser::TEraser(const TClientSet& clients, size_t retries, TDuration timeout)
        : Clients_(clients)
        , Retries_(retries)
        , Timeout_(timeout)
    {
    }

    TString TEraser::Erase(const TString& space, const TString& keys, TInstant instant, time_t expirationTime) const {
        TEraseState state(Clients_.GetAllClients(),
                          BuildRequest(space, keys, instant, expirationTime),
                          Retries_,
                          Timeout_);
        return state.Run(expirationTime);
    }

    TToErasePtr TEraser::BuildRequest(const TString& space,
                                      TStringBuf keys,
                                      TInstant instant,
                                      time_t expirationTime) {
        TToErasePtr res = std::make_shared<kolmogor::replication::v2::ToErase>();

        res->mutable_gen()->set_expiretime(expirationTime);
        res->mutable_gen()->set_space(space);

        res->mutable_data()->set_instant(instant.GetValue());
        while (keys) {
            TStringBuf key = keys.NextTok(',');
            // Пустые строки отфильтрованы ранее при обработке HTTP запроса
            res->mutable_data()->add_key(key.data(), key.size());
        }

        return res;
    }

    TEraseState::TEraseState(const TClientSet::TSet& clients, TToErasePtr data, size_t retries, TDuration timeout)
        : Data_(std::move(data))
        , Timeout_(timeout / retries)
        , Retries_(retries)
    {
        for (const TClientPtr& cl : clients) {
            Clients_.emplace(cl, TClientState{});
        }
    }

    TString TEraseState::Run(time_t expirationTime) {
        grpc::CompletionQueue cq;

        InitRequests(cq);
        bool res = ProcessResponses(cq);

        cq.Shutdown();

        AddDebt(cq, expirationTime);

        return res ? TString() : BuildErrorText();
    }

    void TEraseState::InitRequests(grpc::CompletionQueue& cq) {
        for (auto& [client, state] : Clients_) {
            auto p = client->SendErase(Data_, cq, Timeout_);
            p.release(); // to be deleted with cq
        }
    }

    bool TEraseState::ProcessResponses(grpc::CompletionQueue& cq) {
        size_t succeed = 0;
        const auto deadline =
            std::chrono::system_clock::now() + std::chrono::milliseconds(Timeout_.MilliSeconds());

        while (true) {
            void* tag;
            bool ok;
            switch (cq.AsyncNext(&tag, &ok, deadline)) {
                case grpc::CompletionQueue::SHUTDOWN:
                case grpc::CompletionQueue::TIMEOUT:
                    return false;
                case grpc::CompletionQueue::GOT_EVENT:
                    break;
            }

            try {
                std::unique_ptr<TEraseRequest> req(static_cast<TEraseRequest*>(tag));
                Y_ENSURE(req);

                TString err = req->GetErrorForSyncRequest(ok);
                if (err) {
                    req->Cancel();
                    TClientState& s = GetState(req->GetClient());
                    s.LastError = std::move(err);
                    if (++s.Tries < Retries_) {
                        auto p = req->GetClient().SendErase(Data_, cq, Timeout_);
                        p.release(); // to be deleted with cq
                    }
                    continue;
                }

                TClientState& s = GetState(req->GetClient());
                s.Status = TClientState::EStatus::Ready;
                ++succeed;

                if (Clients_.size() == succeed) {
                    return true;
                }
            } catch (const std::exception& e) {
                TLog::Error() << "Erase: exception: " << e.what();
            }
        }
    }

    void TEraseState::AddDebt(grpc::CompletionQueue& cq, time_t expirationTime) {
        void* tag;
        bool ok;
        while (cq.Next(&tag, &ok)) {
            std::unique_ptr<TEraseRequest> req(static_cast<TEraseRequest*>(tag));
            if (req) {
                req->Cancel();
            }
        }

        for (auto& [client, state] : Clients_) {
            if (state.Status != TClientState::EStatus::Ready) {
                client->SendErase(Data_, expirationTime);
                if (state.LastError.empty()) {
                    state.LastError = "timeout";
                }
            }
        }
    }

    TEraseState::TClientState& TEraseState::GetState(const TClient& c) {
        auto it = Clients_.find(c.Uri());
        Y_VERIFY(it != Clients_.end());
        return it->second;
    }

    TString TEraseState::BuildErrorText() const {
        TStringStream s;
        s.Reserve(256 * Clients_.size());

        for (const auto& [client, state] : Clients_) {
            s << client->Uri()
              << ":" << (state.Status == TClientState::EStatus::Ready ? "ok" : state.LastError)
              << ";";
        }

        return s.Str();
    }
}
