#include "replicator.h"

#include "acceptor.h"
#include "clientset.h"
#include "debt_cleaner.h"
#include "eraser.h"
#include "pinger.h"
#include "server.h"
#include "tailset.h"

namespace NPassport::NKolmogor {
    TReplicator::TReplicator(TMemStorage& storage, const TAuth& auth, const TReplicatorSettings& config)
        : DebtDispatcher_(config.MaxDebtSize ? std::make_shared<NLb::TResourceDispatcher>("replication.debt_bytes") : nullptr)
        , Clients_(ClientCq_,
                   auth,
                   {
                       .Timeout = config.Timeout,
                       .MaxMessageSize = config.MaxMessageSize,
                       .DebtDispatcher = DebtDispatcher_,
                   },
                   config.Uris)
        , Tails_(std::make_unique<TTailSet>(config.GroupCount))
        , Eraser_(std::make_unique<TEraser>(Clients_, config.EraseRetries, config.EraseTimeout))
        , Server_(std::make_unique<TServer>(
              storage,
              auth,
              TServerSettings{
                  .Port = config.Port,
                  .Threads = config.Threads,
                  .MaxMessageSize = config.MaxMessageSize,
              }))
        , Acceptor_(std::make_unique<TAcceptor>(ClientCq_, Clients_, *Tails_))
        , Pinger_(std::make_unique<TPinger>(Clients_, config.PingPeriod))
        , Cleaner_(std::make_unique<TDebtCleaner>(Clients_, config.DebtCleanPeriod))
    {
        if (!DebtDispatcher_) {
            return;
        }

        ui64 numClients = Clients_.GetAllClients().size();
        ui64 clientDebtSize = (numClients ? (config.MaxDebtSize + numClients - 1) / numClients : 0);
        double reserveRatio = 0.2;
        for (const auto& client : Clients_.GetAllClients()) {
            DebtDispatcher_->AddResource(client->HostInSignal(), clientDebtSize, clientDebtSize * reserveRatio);
        }
    }

    TReplicator::~TReplicator() = default;

    void TReplicator::AddUnistat(NUnistat::TBuilder& builder) {
        Clients_.AddUnistat(builder);
        Tails_->AddUnistat(builder);
        Server_->AddUnistat(builder);
        if (DebtDispatcher_) {
            DebtDispatcher_->AddUnistat(builder);
        }
    }

    void TReplicator::AddSpace(const TString& space, ui16 threadCount) {
        Tails_->AddSpace(space, threadCount);
        Server_->AddSpace(space);
    }

    void TReplicator::AddToTail(const TString& space,
                                TStrVec&& keys,
                                TInstant instant,
                                time_t expire) {
        Tails_->AddData(space, std::move(keys), instant, expire);
    }

    TString TReplicator::SendErase(const TString& space,
                                   const TString& keys,
                                   TInstant instant,
                                   time_t expirationTime) {
        return Eraser_->Erase(space, keys, instant, expirationTime);
    }

    void TReplicator::StartWork() {
        Server_->StartWork();
    }

    void TReplicator::StopThreads() {
        Server_->BeginStoping();

        // Send everything
        Tails_.reset();
        // It doesn't matter now
        Pinger_.reset();
        Cleaner_.reset();
        // acceptor stops CompletionQueue - it must be last
        Acceptor_.reset();
        // Stop server
        Server_.reset();
    }
}
