#include "base_request.h"

#include <passport/infra/daemons/kolmogor/src/replication/client.h>

#include <passport/infra/daemons/kolmogor/src/common/auth.h>

#include <passport/infra/libs/cpp/utils/log/global.h>

namespace NPassport::NKolmogor {
    static const std::map<grpc::StatusCode, TString> STATUS_CODE_TO_STRING = {
        {grpc::StatusCode::OK, "OK"},
        {grpc::StatusCode::CANCELLED, "CANCELLED"},
        {grpc::StatusCode::UNKNOWN, "UNKNOWN"},
        {grpc::StatusCode::INVALID_ARGUMENT, "INVALID_ARGUMENT"},
        {grpc::StatusCode::DEADLINE_EXCEEDED, "DEADLINE_EXCEEDED"},
        {grpc::StatusCode::NOT_FOUND, "NOT_FOUND"},
        {grpc::StatusCode::ALREADY_EXISTS, "ALREADY_EXISTS"},
        {grpc::StatusCode::PERMISSION_DENIED, "PERMISSION_DENIED"},
        {grpc::StatusCode::UNAUTHENTICATED, "UNAUTHENTICATED"},
        {grpc::StatusCode::RESOURCE_EXHAUSTED, "RESOURCE_EXHAUSTED"},
        {grpc::StatusCode::FAILED_PRECONDITION, "FAILED_PRECONDITION"},
        {grpc::StatusCode::ABORTED, "ABORTED"},
        {grpc::StatusCode::OUT_OF_RANGE, "OUT_OF_RANGE"},
        {grpc::StatusCode::UNIMPLEMENTED, "UNIMPLEMENTED"},
        {grpc::StatusCode::INTERNAL, "INTERNAL"},
        {grpc::StatusCode::UNAVAILABLE, "UNAVAILABLE"},
        {grpc::StatusCode::DATA_LOSS, "DATA_LOSS"},
    };

    static TStringBuf StatusToString(grpc::StatusCode status) {
        auto it = STATUS_CODE_TO_STRING.find(status);
        return it == STATUS_CODE_TO_STRING.end()
                   ? TStringBuf()
                   : it->second;
    }

    static EClientErrors GrpcStatusToClientErr(grpc::StatusCode status) {
        switch (status) {
            case grpc::StatusCode::DEADLINE_EXCEEDED:
                return EClientErrors::Timeout;
            case grpc::StatusCode::UNAVAILABLE:
                return EClientErrors::NetworkError;
            default:
                return EClientErrors::Other;
        }
    }

    const TString TBaseRequest::X_YA_SERVICE_TICKET = "x-ya-service-ticket"; // https://st.yandex-team.ru/DTCC-695

    TBaseRequest::TBaseRequest(TClient& client, grpc::CompletionQueue& cq)
        : Client_(client)
        , Cq_(cq)
    {
    }

    void TBaseRequest::SetTimeout(TDuration timeout) {
        using namespace std::chrono;
        Ctx_.set_deadline(system_clock::now() + milliseconds(timeout.MilliSeconds()));
    }

    void TBaseRequest::SetAuth(const TAuth& auth) {
        Ctx_.AddMetadata(X_YA_SERVICE_TICKET, auth.GetServiceTicketForReplication());
    }

    bool TBaseRequest::CheckStatus(bool isOk, TString* err) {
        if (isOk && Status_.error_code() == grpc::StatusCode::OK) {
            Client_.MakeOk();
            return true;
        }

        if (!isOk) {
            Client_.MakeDown(EClientErrors::NetworkError);
            TLog::Warning() << "Request failed: " << LogName()
                            << ": " << Client_.Uri()
                            << ": unknown network error";
            if (err) {
                *err = "unknown network error";
            }

            return false;
        }

        Client_.MakeDown(GrpcStatusToClientErr(Status_.error_code()));
        TLog::Warning() << "Request failed: " << LogName()
                        << ": " << Client_.Uri()
                        << "; status_code=" << StatusToString(Status_.error_code())
                        << "; message: " << Status_.error_message()
                        << "; details: " << Status_.error_details();
        if (err) {
            *err = NUtils::CreateStr(
                "status_code=", StatusToString(Status_.error_code()),
                "; message: ", Status_.error_message(),
                "; details: ", Status_.error_details());
        }

        return false;
    }
}
