#include "server.h"

#include "response/erase_response.h"
#include "response/ping_response.h"
#include "response/push_response.h"

#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/unistat/diff.h>
#include <passport/infra/libs/cpp/utils/log/global.h>

#include <contrib/libs/grpc/include/grpcpp/server_builder.h>

#include <util/string/cast.h>
#include <util/system/thread.h>

#include <unordered_map>

namespace NPassport::NKolmogor {
    void TResponserUnistat::AddSpace(const TString& space) {
        PerSpace.emplace(space, std::make_unique<TPerSpace>(space));
    }

    TResponserUnistat::TPerSpace::TPerSpace(const TString& space)
        : IncRequests("replication.space/" + space + "/inc.requests")
        , IncKeys("replication.space/" + space + "/inc.keys")
        , EraseRequests("replication.space/" + space + "/erase.requests")
        , EraseKeys("replication.space/" + space + "/erase.keys")
    {
    }

    TServer::TServer(TMemStorage& storage, const TAuth& auth, const TServerSettings& settings)
        : Service_(std::make_unique<kolmogor::replication::v2::Repl::AsyncService>())
        , Settings_(settings)
        , Storage_(storage)
        , Auth_(auth)
    {
    }

    TServer::~TServer() {
        try {
            if (Server_) {
                Server_->Wait();
            }
            for (auto& t : Pool_) {
                t.T.join();
            }
        } catch (const std::exception&) {
        }
    }

    void TServer::AddSpace(const TString& space) {
        Spaces_.push_back(space);
        Unistat_.AddSpace(space);
    }

    void TServer::StartWork() {
        grpc::ServerBuilder builder;

        builder.SetMaxReceiveMessageSize(Settings_.MaxMessageSize);
        builder.SetMaxSendMessageSize(Settings_.MaxMessageSize);

        builder.AddListeningPort(NUtils::CreateStr("[::]:", Settings_.Port),
                                 grpc::InsecureServerCredentials());
        builder.RegisterService(Service_.get());

        for (size_t idx = 0; idx < Settings_.Threads; ++idx) {
            Pool_.push_back(TThreadItem{
                .Cq = builder.AddCompletionQueue(),
            });
        }

        Server_ = builder.BuildAndStart();
        Y_ENSURE(Server_, "failed to start replication server");

        for (TThreadItem& i : Pool_) {
            i.T = std::thread([this, &i]() {
                TPingResponse::CreateNext(*Service_, *i.Cq, Spaces_);
                TPushResponse::CreateNext(*Service_, *i.Cq, Storage_, Auth_);
                TEraseResponse::CreateNext(*Service_, *i.Cq, Storage_, Auth_);

                void* tag;
                bool ok;

                while (i.Cq->Next(&tag, &ok)) {
                    std::unique_ptr<TBaseResponse> base(static_cast<TBaseResponse*>(tag));

                    if (!ok) {
                        // shutting down
                        continue;
                    }

                    if (base->Run(Unistat_)) {
                        base.release();
                    }
                }
            });
        }
    }

    void TServer::BeginStoping() {
        if (Server_) {
            Server_->Shutdown();
        }
        for (TThreadItem& i : Pool_) {
            i.Cq->Shutdown();
        }
    }

    void TServer::AddUnistat(NUnistat::TBuilder& builder) {
        builder.Add(Unistat_.IncRequests);
        builder.Add(Unistat_.IncKeys);
        builder.Add(Unistat_.EraseRequests);
        builder.Add(Unistat_.EraseKeys);
        builder.Add(Unistat_.Hello);
        builder.Add(Unistat_.Total);

        for (const auto& [name, signals] : Unistat_.PerSpace) {
            builder.Add(signals->IncRequests);
            builder.Add(signals->IncKeys);
            builder.Add(signals->EraseRequests);
            builder.Add(signals->EraseKeys);
        }
    }
}
