#include "tail.h"

#include <passport/infra/daemons/kolmogor/src/common/auth.h>

#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/log/global.h>

#include <util/string/split.h>

namespace NPassport::NKolmogor {
    TTail::TTail(const TString& space, ui16 groupCount, ui16 threadCount)
        : Space(space)
        , GroupCount_(groupCount)
        , UnistatSize_("replication.space/" + space + "/queue.keys", NUnistat::NSuffix::AXXX)
    {
        Y_ENSURE(GroupCount_);
        for (int i = 0; i < threadCount; ++i) {
            Workers_.push_back(std::make_unique<NUtils::TRegularTask>([this]() { Worker(); },
                                                                      TDuration::MilliSeconds(300),
                                                                      "kolmogor_tail"));
        }
    }

    TTail::~TTail() {
        Workers_.clear();
        try {
            Worker();
        } catch (...) { // Sorry
        }
    }

    void TTail::AddUnistat(NUnistat::TBuilder& builder) {
        builder.Add(UnistatSize_);
    }

    void TTail::AddData(TStrVec&& keys, TInstant instant, time_t expire) {
        UnistatSize_ += keys.size();
        Queue_.Enqueue(TNode{std::move(keys), instant, expire});
    }

    void TTail::AddClient(TClientPtr c) {
        std::unique_lock lock(MutexClients_);
        Clients_.insert(std::move(c));
    }

    void TTail::DeleteClient(const TString& uri) {
        std::unique_lock lock(MutexClients_);

        auto it = Clients_.find(uri);
        if (it == Clients_.end()) {
            TLog::Debug() << "Tail: failed to delete client: " << uri << ". space: " << Space;
            return;
        }

        Clients_.erase(it);
    }

    void TTail::Worker() {
        while (PushPart()) {
        }
    }

    std::pair<kolmogor::replication::v2::Tail, time_t> TTail::MakeCurrentPart() {
        TNode data;
        if (!Queue_.Dequeue(&data)) {
            return {};
        }

        size_t keys = data.Keys.size();

        kolmogor::replication::v2::Tail tail;
        AddDataToProto(tail, data);
        time_t expireTime = data.Expire;

        for (size_t idx = 0; idx < GroupCount_ - 1 && Queue_.Dequeue(&data); ++idx) {
            AddDataToProto(tail, data);
            expireTime = std::max(expireTime, data.Expire);
            keys += data.Keys.size();
        }

        UnistatSize_ -= keys;

        tail.mutable_gen()->set_expiretime(expireTime);
        tail.mutable_gen()->set_space(Space);

        return {std::move(tail), expireTime};
    }

    bool TTail::PushPart() {
        auto current = MakeCurrentPart();
        bool isCurrentUseful = !current.first.Getdata().empty();
        TTailPtr ptr;
        if (isCurrentUseful) {
            ptr = std::make_shared<kolmogor::replication::v2::Tail>(std::move(current.first));
        }

        bool res = false;

        std::shared_lock lock(MutexClients_);
        for (const TClientPtr& c : Clients_) {
            res = res || c->SendSomeDebt();
            if (isCurrentUseful) {
                c->SendPush(ptr, current.second);
            }
        }

        return res || isCurrentUseful;
    }

    static kolmogor::replication::v2::Data* GetCurData(
        TInstant cur,
        kolmogor::replication::v2::Tail& req,
        int rIdx) {
        int idx = req.data_size() - rIdx - 1;
        if (idx >= 0 && idx < req.data_size() && TInstant::FromValue(req.data(idx).instant()) == cur) {
            return req.mutable_data(idx);
        }
        return nullptr;
    }

    void TTail::AddDataToProto(kolmogor::replication::v2::Tail& req, const TNode& data) {
        kolmogor::replication::v2::Data* d = GetCurData(data.Instant, req, 0);
        if (!d) {
            d = GetCurData(data.Instant, req, 1);
        }
        if (!d) {
            d = req.add_data();
            d->set_instant(data.Instant.GetValue());
        }

        for (const TString& key : data.Keys) {
            d->add_key(key);
        }
    }
}
