#include "splited_counter.h"

#include <util/stream/output.h>
#include <util/string/cast.h>
#include <util/string/split.h>

#include <algorithm>
#include <ctime>

namespace NPassport::NKolmogor {
    /*
     * Счетчик для текущего периода всегда в конце массива.
     * Если в конце не текущий, то значит, он устарел.
     * Чистим массив только тогда, когда места уже нет.
     */

    TSplitedCounter::TSplitedCounter(const TCounterParams& params)
        : Params_(params)
    {
        Data_.reserve(Params_.CounterCount);
    }

    TSplitedCounter TSplitedCounter::FromProto(const TCounterParams& params,
                                               const kolmogor_persistency::Slice_Pair::SplitedCounter& key) {
        TSplitedCounter res(params);

        for (int idx = 0; idx < key.piece_size(); ++idx) {
            res.Data_.push_back({params.FromInstant(TInstant::FromValue(key.piece(idx).instant())),
                                 key.piece(idx).value()});
        }

        res.ErasedUpToInstant_ = TInstant::FromValue(key.eraseinstant());
        res.ErasedUpToPeriod_ = params.FromInstant(res.ErasedUpToInstant_);

        return res;
    }

    void TSplitedCounter::ToProto(kolmogor_persistency::Slice::Pair::SplitedCounter& out) const {
        auto& pieces = *out.mutable_piece();
        pieces.Reserve(Data_.size());
        for (const TPiece& piece : Data_) {
            kolmogor_persistency::Slice_Pair_SplitedCounter_Piece* p = pieces.Add();
            p->set_instant(Params_.ToInstantBegining(piece.PeriodNo).GetValue());
            p->set_value(piece.Counter);
        }

        out.set_eraseinstant(ErasedUpToInstant_.GetValue());
    }

    ui64 TSplitedCounter::Get(const TKolmoPeriod cur) const {
        ui64 res = 0;

        for (auto it = Data_.rbegin(); it != Data_.rend(); ++it) {
            if (!IsValid(it->PeriodNo, cur)) {
                break;
            }
            res += it->Counter;
        }

        return res;
    }

    void TSplitedCounter::Inc(TInstant instant, const TKolmoPeriod cur, ui32 value) {
        if (instant <= ErasedUpToInstant_) {
            return;
        }

        TKolmoPeriod period = Params_.FromInstant(instant);
        if (Params_.IsExpired(period, cur)) {
            return;
        }

        for (auto it = Data_.rbegin(); it != Data_.rend(); ++it) {
            if (Params_.IsExpired(it->PeriodNo, cur) || it->PeriodNo < period) {
                break;
            }
            if (it->PeriodNo == period) {
                // it is in array
                it->Counter += value;
                return;
            }
        }

        if (Data_.size() >= Params_.CounterCount) {
            // need clean up
            CleanExpired(cur);
        }

        auto it = std::find_if(Data_.begin(),
                               Data_.end(),
                               [period](const TPiece& p) { return period < p.PeriodNo; });
        Data_.insert(it,
                     TPiece{
                         .PeriodNo = period,
                         .Counter = value,
                     });
    }

    void TSplitedCounter::Erase(TInstant instant) {
        if (instant <= ErasedUpToInstant_) {
            return;
        }

        ErasedUpToInstant_ = instant;
        ErasedUpToPeriod_ = Params_.FromInstant(ErasedUpToInstant_);

        for (auto it = Data_.rbegin(); it != Data_.rend(); ++it) {
            if (it->PeriodNo == ErasedUpToPeriod_) {
                it->Counter = 0;
                break;
            }

            if (it->PeriodNo < ErasedUpToPeriod_) {
                break;
            }
        }
    }

    bool TSplitedCounter::NeedClean(const TKolmoPeriod cur) const {
        return Params_.IsExpired(ErasedUpToPeriod_, cur) &&
               (Data_.empty() || Params_.IsExpired(Data_.back().PeriodNo, cur));
    }

    TString TSplitedCounter::ToDebugString() const {
        TStringStream res;

        for (const TPiece& p : Data_) {
            res << p.PeriodNo.Val << ":" << p.Counter << ";";
        }

        return res.Str();
    }

    void TSplitedCounter::CleanExpired(const TKolmoPeriod cur) {
        auto it = std::find_if(Data_.begin(),
                               Data_.end(),
                               [this, cur](const TPiece& p) { return IsValid(p.PeriodNo, cur); });
        Data_.erase(Data_.begin(), it);
    }

    bool TSplitedCounter::IsValid(TKolmoPeriod period, const TKolmoPeriod cur) const {
        return !Params_.IsExpired(period, cur) && ErasedUpToPeriod_ <= period;
    }
}
