#include "mem_storage.h"

#include <passport/infra/daemons/kolmogor/src/common/exception.h>
#include <passport/infra/daemons/kolmogor/src/replication/replicator.h>

#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/log/global.h>

#include <util/generic/string.h>
#include <util/stream/output.h>
#include <util/string/builder.h>
#include <util/string/cast.h>
#include <util/system/fs.h>
#include <util/system/user.h>
#include <util/thread/pool.h>

namespace NPassport::NKolmogor {
    TMemStorage::TMemStorage()
        : Unistat_(0)
    {
    }

    TMemStorage::TMemStorage(const TMemStorageSettings& settings)
        : Unistat_(0)
        , Settings_(settings)
    {
        Stor_.max_load_factor(0.5);
        Stor_.reserve(16);
    }

    TMemStorage::~TMemStorage() = default;

    void TMemStorage::AddSpace(const TSpaceSettings& conf) {
        TLog::Info() << "Adding space: " << conf.Name;
        auto s = std::make_unique<TSpace>(conf, Settings_.DataDirectory, Settings_.CleanCount);

        Y_ENSURE(Stor_.emplace(conf.Name, std::move(s)).second);
    }

    NV2::TIncResult TMemStorage::IncFromHttp(NV2::TIncRequest&& req) {
        NV2::TIncResult res;
        res.Data.reserve(req.Req.size());

        for (const NV2::TIncRequest::TSpace& space : req.Req) {
            if (Stor_.find(space.Name) == Stor_.end()) {
                throw TBadRequestException() << "Space is not found (inc): " << space.Name;
            }
        }

        const TInstant now = TInstant::Now();

        for (NV2::TIncRequest::TSpace& inSpace : req.Req) {
            auto it = Stor_.find(inSpace.Name);
            Y_VERIFY(it != Stor_.end());
            TSpace& space = *it->second;

            res.Data.push_back(NV2::TIncResult::TSpace{
                .Name = inSpace.Name,
                .Values = space.IncAndGet(inSpace, now),
            });

            TStrVec toRepl;
            toRepl.reserve(inSpace.KeysCount);
            // result does not have enough info about usual keys, so we'll get info from request
            for (const NV2::TIncRequest::TKeySet& s : inSpace.Keysets) {
                if (!s.IncIfLessThan) {
                    toRepl.insert(toRepl.end(), s.Keys.begin(), s.Keys.end());
                }
            }
            // result shows actual diff for conditionable values
            for (const NV2::TIncResult::TKeyValue& s : res.Data.back().Values) {
                if (!s.Diff) {
                    continue;
                }
                for (size_t idx = 0; idx < *s.Diff; ++idx) {
                    toRepl.push_back(s.Key);
                }
            }

            SendReplication(inSpace.Name,
                            std::move(toRepl),
                            now,
                            space.Params().ExpireTime(now));
        }

        return res;
    }

    void TMemStorage::Inc(const TString& space, const TStrVec& keys, TInstant instant) {
        auto it = Stor_.find(space);
        if (it == Stor_.end()) { // Already checked before. Impossible
            return;
        }

        it->second->Inc(keys, instant);
    }

    NV2::TGetResult TMemStorage::Get(const NV2::TGetRequest& req, TSpace::EAPI apiVer) const {
        NV2::TGetResult res;
        res.Data.reserve(req.Req.size());

        for (const NV2::TGetRequest::TSpace& inSpace : req.Req) {
            auto it = Stor_.find(inSpace.Name);
            if (it == Stor_.end()) {
                throw TBadRequestException() << "Space is not found (get): " << inSpace.Name;
            }
            TSpace& space = *it->second;

            res.Data.push_back(NV2::TGetResult::TSpace{
                .Name = inSpace.Name,
                .Values = space.Get(inSpace.Keys, apiVer),
            });
        }

        return res;
    }

    TString TMemStorage::Erase(const TString& space, const TString& rawKeys) {
        auto it = Stor_.find(space);
        Y_ENSURE(it != Stor_.end(), "Space is not found (erase): " << space); // Already checked before

        TInstant now = it->second->Erase(rawKeys);
        return SendReplicationErase(space, rawKeys, now, it->second->Params().ExpireTime(now));
    }

    void TMemStorage::Erase(const TString& space, const TStrVec& keys, TInstant instant) {
        auto it = Stor_.find(space);
        if (it == Stor_.end()) { // Already checked before. Impossible
            return;
        }

        it->second->Erase(keys, instant);
    }

    NV2::TEraseAllResult TMemStorage::EraseAll(const NV2::TEraseAllRequest& req) {
        auto it = Stor_.find(req.SpaceName);
        if (it == Stor_.end()) {
            throw TBadRequestException() << "Space is not found (eraseall): " << req.SpaceName;
        }

        NV2::TEraseAllResult res;
        res.SpaceName = req.SpaceName;
        res.DeletedKeys = it->second->EraseAll();
        return res;
    }

    void TMemStorage::AddUnistat(NUnistat::TBuilder& builder) {
        TSpace::TStats s;
        for (const auto& [name, space] : Stor_) {
            s += space->AddUnistat(builder);
        }

        builder.AddRow(Unistat_.GetRequests.GetName(), s.GetRequests);
        builder.AddRow(Unistat_.GetKeys.GetName(), s.GetKeys);
        builder.AddRow(Unistat_.IncRequests.GetName(), s.IncRequests);
        builder.AddRow(Unistat_.IncKeys.GetName(), s.IncKeys);
        builder.AddRow(Unistat_.EraseRequests.GetName(), s.EraseRequests);
        builder.AddRow(Unistat_.EraseKeys.GetName(), s.EraseKeys);

        builder.AddRow(Unistat_.Space->Keys().GetName(), s.Keys);
        builder.AddRow(Unistat_.Space->Memory().GetName(), s.Memory);
    }

    bool TMemStorage::HasSpace(const TString& space) const {
        return Stor_.find(space) != Stor_.end();
    }

    void TMemStorage::SetReplicator(std::weak_ptr<TReplicator> replicator) {
        Replicator_ = replicator;
    }

    void TMemStorage::SendReplication(const TString& space,
                                      TStrVec&& keys,
                                      TInstant instant,
                                      time_t expire) {
        std::shared_ptr<TReplicator> replicator = Replicator_.lock();
        if (replicator) {
            replicator->AddToTail(space, std::move(keys), instant, expire);
        }
    }

    TString TMemStorage::SendReplicationErase(const TString& space,
                                              const TString& keys,
                                              TInstant instant,
                                              time_t expirationTime) {
        std::shared_ptr<TReplicator> replicator = Replicator_.lock();
        if (replicator) {
            try {
                return replicator->SendErase(space, keys, instant, expirationTime);
            } catch (const std::exception& e) {
                return TStringBuilder() << "Failed to erase data on other instances: " << e.what();
            }
        }
        return {};
    }

    void TMemStorage::StartCleaning() {
        ui32 min = std::numeric_limits<ui32>::max();
        for (const auto& p : Stor_) {
            min = std::min(min, p.second->Params().ActualDuration);
        }

        TDuration period = TDuration::Seconds(min);
        if (period < Settings_.MinCleanPeriod) {
            period = Settings_.MinCleanPeriod;
        }

        Cleaner_ = std::make_unique<TCleaner>(
            [this]() { Cleaner(); },
            period,
            "kolmogor_clean");
    }

    void TMemStorage::StopThreads() {
        Cleaner_.reset();
    }

    void TMemStorage::LoadOnStart() {
        EnsurePath();

        TThreadPool queue;
        queue.Start(Settings_.ThreadsForDump);

        TLog::Info() << "Start of loading from disk with " << Settings_.ThreadsForDump << " threads";
        const TInstant start = TInstant::Now();

        for (auto& p : Stor_) {
            p.second->Load(queue);
        }

        queue.Stop();

        const TInstant end = TInstant::Now();
        TLog::Info() << "Loading from disk took " << (end - start);
    }

    void TMemStorage::SaveOnExit() {
        Y_ENSURE(Replicator_.expired(), "replicator cannot be properly released as it is owned somewhere else");
        EnsurePath();

        TThreadPool queue;
        queue.Start(Settings_.ThreadsForDump);

        TLog::Info() << "Start of saving to disk with " << Settings_.ThreadsForDump << " threads";
        const TInstant start = TInstant::Now();

        for (const auto& p : Stor_) {
            TLog::Debug() << "started flushing space " << p.first;
            p.second->Save(queue);
            TLog::Debug() << "space flushed: " << p.first;
        }

        queue.Stop();

        const TInstant end = TInstant::Now();
        TLog::Info() << "Saving to disk took " << (end - start);
    }

    void TMemStorage::Cleaner() {
        for (auto& p : Stor_) {
            p.second->Clean();
        }
    }

    void TMemStorage::EnsurePath() {
        Y_ENSURE(
            NFs::MakeDirectoryRecursive(Settings_.DataDirectory),
            "user " << GetUsername() << " cannot create "
                    << Settings_.DataDirectory << " [" << errno << "]\n");
    }
}
