#include "slice.h"

#include <passport/infra/daemons/kolmogor/src/common/utils.h>

#include <passport/infra/libs/cpp/utils/log/global.h>

#include <util/generic/strbuf.h>
#include <util/stream/file.h>
#include <util/string/cast.h>
#include <util/string/split.h>
#include <util/system/fs.h>
#include <util/thread/pool.h>

namespace NPassport::NKolmogor {
    TSpaceSizeStats::TSpaceSizeStats(const TString& id, std::optional<size_t> memoryLimit)
        : Keys_("common." + id + "keys", NUnistat::NSuffix::AXXX)
        , Memory_("common." + id + "memory", NUnistat::NSuffix::AXXX)
        , Limited_("common." + id + "keys_limited")
        , MemoryLimit_(memoryLimit)
    {
    }

    bool TSpaceSizeStats::TryAdd(size_t size) {
        if (MemoryLimit_ && *MemoryLimit_ < Memory_.GetValue() + size) {
            return false;
        }

        Memory_ += size;
        ++Keys_;

        return true;
    }

    void TSpaceSizeStats::Erase(size_t size) {
        Memory_ -= size;
        --Keys_;
    }

    TSlice::TSlice(size_t reserve,
                   const TCounterParams& params,
                   size_t cleanCount,
                   size_t eraseCount,
                   TSpaceSizeStatsPtr stats)
        : Mutex_(std::make_unique<std::shared_timed_mutex>())
        , Params_(params)
        , MemoryPerKey_(sizeof(TSplitedCounter) + params.CounterCount * sizeof(TSplitedCounter::TPiece))
        , CleanCount_(cleanCount)
        , EraseCount_(eraseCount)
        , Stats_(std::move(stats))
    {
        Y_ENSURE(Params_.ActualDuration != 0, "Invalid config: ttl > count");

        Data_.max_load_factor(0.5);
        Data_.reserve(reserve);
    }

    bool TSlice::Inc(const TStrVec& keys, const TIdxVec& idxs, TInstant instant, const TKolmoPeriod cur) {
        std::unique_lock lock(*Mutex_);

        bool wasLimited = false;
        for (size_t idx : idxs) {
            TSplitedCounter* sc = FindOrInsertNode(keys[idx]);
            if (sc) {
                sc->Inc(instant, cur);
            } else {
                wasLimited = true;
            }
        }

        return !wasLimited;
    }

    bool TSlice::IncAndGet(const TIdxVec& idxs,
                           const TInstant instant,
                           const TKolmoPeriod cur,
                           NV2::TIncResult::TValues& out) {
        std::unique_lock lock(*Mutex_);
        bool wasLimited = false;

        // `out` is actually `in` too: it is the same struct just to reduce memory allocations

        for (size_t idx : idxs) {
            TSplitedCounter* sc = FindOrInsertNode(out[idx].Key);
            if (!sc) {
                out[idx].Value = 0;
                if (out[idx].Diff) {
                    out[idx].Diff = 0;
                }
                wasLimited = true;
                continue;
            }

            const std::optional<ui64> incIfLess = out[idx].Diff;
            ui64 diff = out[idx].Value;

            const ui64 prevValue = sc->Get(cur);
            if (incIfLess) {
                diff = prevValue < *incIfLess
                           ? std::min(diff, *incIfLess - prevValue)
                           : 0;
                out[idx].Diff = diff;
            }

            sc->Inc(instant, cur, diff);
            out[idx].Value = prevValue + diff;
        }

        return !wasLimited;
    }

    void TSlice::Get(const TIdxVec& idxs, const TKolmoPeriod cur, NV2::TGetResult::TValues& out) const {
        std::shared_lock lock(*Mutex_);

        for (size_t idx : idxs) {
            const TString& k = out[idx].Key;
            auto it = Data_.find(k);
            if (it == Data_.end()) {
                continue;
            }

            out[idx].Value = it->second.Get(cur);
        }
    }

    bool TSlice::Erase(const TStrVec& keys, const TIdxVec& idxs, const TInstant instant) {
        std::unique_lock lock(*Mutex_);

        bool wasLimited = false;
        for (size_t idx : idxs) {
            TSplitedCounter* sc = FindOrInsertNode(keys[idx]);
            if (sc) {
                sc->Erase(instant);
            } else {
                wasLimited = true;
            }
        }

        return !wasLimited;
    }

    TSlice::TCleanInfo TSlice::Clean(TKolmoPeriod cur, size_t idx) {
        std::unique_lock lock(*Mutex_);
        size_t deleted = 0;
        TInstant start = TInstant::Now();

        // Чистка разбита на участки, чтобы не блокировать Slice на слишком длительное время
        for (size_t count = 0; count < CleanCount_ && idx < Data_.bucket_count(); ++idx, ++count) {
            for (auto it = Data_.begin(idx); it != Data_.end(idx);) {
                if (it->second.NeedClean(cur)) {
                    Stats_->Erase(MemoryPerKey_ + it->first.size());
                    Data_.erase((it++)->first); // TODO use extract
                    ++deleted;
                } else {
                    ++it;
                }
            }
        }

        TInstant end = TInstant::Now();
        return {idx < Data_.bucket_count(),
                idx,
                deleted,
                end - start};
    }

    TSlice::TCleanInfo TSlice::EraseAll(TInstant instant, size_t idx) {
        std::unique_lock lock(*Mutex_);
        size_t deleted = 0;
        TInstant start = TInstant::Now();

        for (size_t count = 0; count < EraseCount_ && idx < Data_.bucket_count(); ++idx, ++count) {
            for (auto it = Data_.begin(idx); it != Data_.end(idx); it++) {
                it->second.Erase(instant);
                deleted++;
            }
        }

        TInstant end = TInstant::Now();
        return {idx < Data_.bucket_count(),
                idx,
                deleted,
                end - start};
    }

    void TSlice::Load(TThreadPool& queue, const TString& fileInputName) {
        if (!NFs::Exists(fileInputName)) {
            return;
        }

        queue.SafeAddFunc([this, fileInputName]() {
            TFileInput fileInput(fileInputName);
            kolmogor_persistency::Slice proto;
            if (!proto.ParseFromString(fileInput.ReadAll())) {
                TLog::Error() << "Failed to parse protobuf: " << fileInputName;
                return;
            }

            std::unique_lock lock(*Mutex_);

            Data_.reserve(proto.counters_size());
            for (int idx = 0; idx < proto.counters_size(); ++idx) {
                if (!Stats_->TryAdd(MemoryPerKey_ + proto.counters(idx).key().size())) {
                    continue;
                }

                auto p = Data_.emplace(proto.counters(idx).key(),
                                       TSplitedCounter::FromProto(Params_, proto.counters(idx).counter()));
                if (!p.second) {
                    TLog::Error() << "Duplicated data in " << fileInputName;
                }
            }
        });
    }

    void TSlice::Save(TThreadPool& queue, const TString& fileOutputName) const {
        if (Data_.empty()) {
            return;
        }

        queue.SafeAddFunc([this, fileOutputName]() {
            google::protobuf::Arena arena;
            kolmogor_persistency::Slice& proto =
                *google::protobuf::Arena::CreateMessage<kolmogor_persistency::Slice>(&arena);
            {
                std::unique_lock lock(*Mutex_);

                auto& counters = *proto.mutable_counters();
                counters.Reserve(Data_.size());
                for (const auto& pair : Data_) {
                    kolmogor_persistency::Slice::Pair* p = counters.Add();
                    p->set_key(pair.first);
                    pair.second.ToProto(*p->mutable_counter());
                }
            }

            TFileOutput fileOutput(fileOutputName);
            fileOutput << proto.SerializeAsString();
        });
    }

    TSplitedCounter* TSlice::FindOrInsertNode(const TString& key) {
        auto [it, inserted] = Data_.try_emplace(key, Params_);
        if (inserted && !Stats_->TryAdd(MemoryPerKey_ + key.size())) {
            // assume it is rare case: it is better to keep fast other branches
            Data_.erase(it);
            return nullptr;
        }

        return &it->second;
    }
}
