#include "space.h"

#include <passport/infra/daemons/kolmogor/src/common/utils.h>
#include <passport/infra/daemons/kolmogor/src/replication/replicator.h>

#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/log/global.h>

#include <util/stream/file.h>
#include <util/string/cast.h>
#include <util/system/fs.h>
#include <util/system/user.h>

#include <algorithm>

namespace NPassport::NKolmogor {
    TSpaceUnistat::TSpaceUnistat(const TString& name, std::optional<size_t> memoryLimit)
        : TSpaceUnistat(0, "space/" + name + "/", memoryLimit)
    {
    }

    TSpaceUnistat::TSpaceUnistat(int)
        : TSpaceUnistat(0, "total.")
    {
    }

    TSpaceUnistat::TSpaceUnistat(int, const TString& id, std::optional<size_t> memoryLimit)
        : GetRequests("common." + id + "get.requests")
        , GetKeys("common." + id + "get.keys")
        , IncRequests("common." + id + "inc.requests")
        , IncKeys("common." + id + "inc.keys")
        , EraseRequests("common." + id + "erase.requests")
        , EraseKeys("common." + id + "erase.keys")
        , Space(std::make_shared<TSpaceSizeStats>(id, memoryLimit))
    {
    }

    TSpace::TSpace(const TSpaceSettings& conf, const TString& dataDirectory, size_t cleanCount)
        : Params_(conf.CounterTtl, conf.CounterCount)
        , Name_(conf.Name)
        , DataDirectory_(dataDirectory)
        , SpaceDataDirectory_(DataDirectory_ + "/" + Name_)
        , Persistency_(conf.Persistency)
        , LastCleanedPeriod_(Params_.CurrentPeriod())
        , Unistat_(conf.Name, conf.MemoryLimit)
        , UnistatCleanDuration_("common.space/" + conf.Name + "/clean.duration",
                                NUnistat::TTimeStat::CreateBoundsFromMaxValue(TDuration::Seconds(10)))
    {
        Arr_.reserve(conf.SliceCount);
        size_t r = std::max(conf.Reserve / conf.SliceCount, 16UL);
        for (size_t idx = 0; idx < conf.SliceCount; ++idx) {
            Arr_.emplace_back(r, Params_, cleanCount, conf.EraseCount, Unistat_.Space);
        }

        TLog::Info() << "Created new space: " << Name_
                     << " (ttl=" << Params_.CounterTtl
                     << ", count=" << Params_.CounterCount << ")";
    }

    TSpace::~TSpace() {
        TLog::Info() << "Deleted space: " << Name_
                     << " (ttl=" << Params_.CounterTtl
                     << ", count=" << Params_.CounterCount << ")";
    }

    NV2::TIncResult::TValues TSpace::IncAndGet(const NV2::TIncRequest::TSpace& keys, TInstant now) {
        Unistat_.IncKeys += keys.KeysCount;
        ++Unistat_.IncRequests;

        struct TPerKey {
            ui32 Num = 0;
            std::optional<ui64> IncIfLessThan;
        };

        std::unordered_map<TString, TPerKey> tmp;
        tmp.reserve(keys.KeysCount);

        for (const NV2::TIncRequest::TKeySet& set : keys.Keysets) {
            for (const TString& k : set.Keys) {
                TPerKey& perKey = tmp.try_emplace(k).first->second;
                ++perKey.Num;
                perKey.IncIfLessThan = set.IncIfLessThan;
            }
        }

        NV2::TIncResult::TValues out;
        out.reserve(tmp.size());
        for (const auto& [key, perKey] : tmp) {
            out.push_back(NV2::TIncResult::TKeyValue{
                .Key = key,
                .Value = perKey.Num,
                .Diff = perKey.IncIfLessThan,
            });
        }

        bool wasLimited = false;
        TKolmoPeriod cur = Params_.FromInstant(now);
        for (const auto& p : GetIdxs(out)) {
            wasLimited = !Arr_[p.first].IncAndGet(p.second, now, cur, out) || wasLimited;
        }
        if (wasLimited) {
            TLog::Warning() << "Space '" << Name_ << "': some keys were not added because of memory limit: from API";
        }

        return out;
    }

    void TSpace::Inc(const TStrVec& keys, TInstant instant) {
        TKolmoPeriod cur = Params_.CurrentPeriod();
        if (Params_.IsExpired(Params_.FromInstant(instant), cur)) { // optimization
            TLog::Debug() << "Space: data from replication is expired";
            return;
        }

        bool wasLimited = false;
        for (const auto& p : GetIdxs(keys)) {
            wasLimited = !Arr_[p.first].Inc(keys, p.second, instant, cur) || wasLimited;
        }
        if (wasLimited) {
            TLog::Warning() << "Space '" << Name_ << "': some keys were not added because of memory limit: from replication";
        }
    }

    NV2::TGetResult::TValues TSpace::Get(const TStrVec& keys, EAPI apiVer) const {
        Unistat_.GetKeys += keys.size();
        ++Unistat_.GetRequests;

        /*
         * В каждом Slice для найденых ключей в нужную позицию выставляется значение счетчика.
         * В каждый необходимый Slice заходим только 1 раз.
         */
        NV2::TGetResult::TValues res;
        res.reserve(keys.size());
        for (const TString& key : keys) {
            res.push_back(NV2::TGetResult::TKeyValue{
                .Key = key,
                .Value = 0,
            });
        }
        if (apiVer == EAPI::V2) {
            std::sort(res.begin(), res.end(), [](const auto& l, const auto& r) { return l.Key < r.Key; });
            res.erase(std::unique(res.begin(), res.end()), res.end());
        }

        TKolmoPeriod cur = Params_.CurrentPeriod();
        for (const auto& p : GetIdxs(res)) {
            Arr_[p.first].Get(p.second, cur, res);
        }

        return res;
    }

    TInstant TSpace::Erase(const TString& rawKeys) {
        TStrVec keys;
        TUtils::Split(rawKeys, ',', keys);

        Unistat_.EraseKeys += keys.size();
        ++Unistat_.EraseRequests;

        TInstant now = TInstant::Now();
        EraseImpl(keys, now);
        return now;
    }

    void TSpace::Erase(const TStrVec& keys, TInstant instant) {
        EraseImpl(keys, instant);
    }

    ui32 TSpace::EraseAll() {
        const TInstant start = TInstant::Now();
        std::vector<size_t> idxs(Arr_.size(), 0);
        TSlice::TCleanInfo info{}; //информация здесь такая же, как и в clean, поэтому нет смысла писать новую структуру
        size_t iterations = 0;
        const TInstant now = TInstant::Now();

        TLog::Info() << "Erasing of '" << Name_ << "' started at " << start;
        do {
            info.IsDirty = false;
            for (int idx = Arr_.size() - 1; idx >= 0; idx--) {
                size_t& i = idxs[idx];
                if (i == std::numeric_limits<size_t>::max()) {
                    continue;
                }
                TSlice::TCleanInfo r = Arr_[idx].EraseAll(now, i);
                info.IsDirty |= r.IsDirty;
                i = r.IsDirty ? r.LastIdx : std::numeric_limits<size_t>::max();
                info.Deleted += r.Deleted;
                info.Duration = std::max(info.Duration, r.Duration);
            }
            iterations++;
        } while (info.IsDirty);

        const TInstant stop = TInstant::Now();

        TLog::Info() << "Erasing of '" << Name_ << "' took " << (stop - start)
                     << " with " << iterations << " iterations,"
                     << " deleted " << info.Deleted
                     << ", longest lock " << info.Duration;

        Unistat_.EraseKeys += info.Deleted;
        ++Unistat_.EraseRequests;

        return info.Deleted;
    }

    void TSpace::Clean() {
        const TKolmoPeriod cur = Params_.CurrentPeriod();
        if (cur <= LastCleanedPeriod_) {
            return;
        }

        const TInstant start = TInstant::Now();

        std::vector<size_t> idxs(Arr_.size(), 0);
        TSlice::TCleanInfo info{};
        size_t iterations = 0;
        // Обратный порядок нужен для того, чтобы каждый внешний запрос попал под раздачу как можно меньше раз
        do {
            info.IsDirty = false;
            for (int idx = Arr_.size() - 1; idx >= 0; --idx) {
                size_t& i = idxs[idx];
                if (i == std::numeric_limits<size_t>::max()) {
                    continue;
                }
                TSlice::TCleanInfo r = Arr_[idx].Clean(cur, i);
                info.IsDirty |= r.IsDirty;
                i = r.IsDirty ? r.LastIdx : std::numeric_limits<size_t>::max();
                info.Deleted += r.Deleted;
                info.Duration = std::max(info.Duration, r.Duration);
            }
            ++iterations;
        } while (info.IsDirty);

        const TInstant stop = TInstant::Now();

        UnistatCleanDuration_.Insert(stop - start);

        LastCleanedPeriod_ = cur;
        TLog::Info() << "Cleaning of '" << Name_ << "' took " << (stop - start)
                     << " with " << iterations << " iterations,"
                     << " deleted " << info.Deleted << ", longest lock "
                     << info.Duration << ", period " << cur.Val;
    }

    TSpace::TStats TSpace::AddUnistat(NUnistat::TBuilder& builder) {
        builder.Add(Unistat_.GetRequests);
        builder.Add(Unistat_.GetKeys);
        builder.Add(Unistat_.IncRequests);
        builder.Add(Unistat_.IncKeys);
        builder.Add(Unistat_.EraseRequests);
        builder.Add(Unistat_.EraseKeys);

        builder.Add(Unistat_.Space->Keys());
        builder.Add(Unistat_.Space->Memory());
        builder.Add(Unistat_.Space->Limited());

        UnistatCleanDuration_.AddUnistat(builder);

        return TStats{
            .GetRequests = Unistat_.GetRequests.GetValue(),
            .GetKeys = Unistat_.GetKeys.GetValue(),
            .IncRequests = Unistat_.IncRequests.GetValue(),
            .IncKeys = Unistat_.IncKeys.GetValue(),
            .EraseRequests = Unistat_.EraseRequests.GetValue(),
            .EraseKeys = Unistat_.EraseKeys.GetValue(),
            .Keys = Unistat_.Space->Keys().GetValue(),
            .Memory = Unistat_.Space->Memory().GetValue(),
        };
    }

    void TSpace::Load(TThreadPool& queue) {
        if (!Persistency_) {
            return;
        }
        EnsurePath();
        ValidateConfigConsistency();

        for (size_t i = 0; i < Arr_.size(); ++i) {
            Arr_[i].Load(queue, SpaceDataDirectory_ + '/' + IntToString<10>(i));
        }
    }

    void TSpace::Save(TThreadPool& queue) const {
        if (!Persistency_) {
            return;
        }

        ClearPath();
        EnsurePath();
        SaveConfig();

        for (size_t i = 0; i < Arr_.size(); ++i) {
            Arr_[i].Save(queue, SpaceDataDirectory_ + '/' + IntToString<10>(i));
        }
    }

    void TSpace::EraseImpl(const TStrVec& keys, TInstant ins) {
        bool wasLimited = false;
        for (const auto& p : GetIdxs(keys)) {
            wasLimited = !Arr_[p.first].Erase(keys, p.second, ins) || wasLimited;
        }
        if (wasLimited) {
            TLog::Warning() << "Space '" << Name_ << "': some keys were not added because of memory limit: erase";
        }
    }

    static const TString& GetKey(const TStrVec& keys, size_t i) {
        return keys[i];
    }

    static const TString& GetKey(const NV2::TIncResult::TValues& keys, size_t i) {
        return keys[i].Key;
    }

    template <class T>
    TSpace::TIdxs TSpace::GetIdxs(const T& keys) const {
        // Важна сортировка: чтобы натолкнуться на cleaner() не более одного раза
        TIdxs idxMap;

        for (size_t i = 0; i < keys.size(); ++i) {
            size_t idx = GetBucketIdx(GetKey(keys, i));

            auto it = idxMap.insert({idx, {}}).first;
            it->second.push_back(i);
        }

        return idxMap;
    }

    size_t TSpace::GetBucketIdx(const TString& s) const {
        return (THash<TString>{}(s) >> sizeof(size_t) * 8 / 2) % Arr_.size();
    }

    void TSpace::ClearPath() const {
        try {
            NFs::RemoveRecursive(SpaceDataDirectory_);
        } catch (const std::exception& ex) {
            TLog::Warning() << "Failed to clear out the existing directory of stored space " << Name_
                            << " : " << ex.what();
        }
    }

    void TSpace::EnsurePath() const {
        Y_ENSURE(
            NFs::MakeDirectoryRecursive(SpaceDataDirectory_),
            "user " << GetUsername() << " cannot create "
                    << SpaceDataDirectory_.c_str() << " [" << errno << "]\n");
    }

    bool TSpace::IsConfigConsistent() const {
        ui64 count = 0;
        try {
            TFileInput fileInput(SpaceDataDirectory_ + "/metadata");
            fileInput >> count;
            return Arr_.size() == count;
        } catch (const std::exception&) {
            return false;
        }
    }

    void TSpace::ValidateConfigConsistency() const {
        if (!IsConfigConsistent()) {
            TLog::Warning() << "Configuration of stored space " << Name_
                            << " is corrupted or inconsistent with current configuration";
            NFs::RemoveRecursive(SpaceDataDirectory_);
            NFs::MakeDirectoryRecursive(SpaceDataDirectory_);
        }
    }

    void TSpace::SaveConfig() const {
        EnsurePath();
        TFixedBufferFileOutput fileOutput(SpaceDataDirectory_ + "/metadata");
        fileOutput << Arr_.size();
    }

    void TSpace::TStats::operator+=(const TSpace::TStats& o) {
        GetRequests += o.GetRequests;
        GetKeys += o.GetKeys;
        IncRequests += o.IncRequests;
        IncKeys += o.IncKeys;
        EraseRequests += o.EraseRequests;
        EraseKeys += o.EraseKeys;

        Keys += o.Keys;
        Memory += o.Memory;
    }
}
