#pragma once

#include "slice.h"

#include <passport/infra/daemons/kolmogor/src/storage/counter/counter_params.h>

#include <passport/infra/daemons/kolmogor/src/common/type.h>
#include <passport/infra/daemons/kolmogor/src/input/inc_request.h>
#include <passport/infra/daemons/kolmogor/src/output/get_result.h>

#include <passport/infra/libs/cpp/unistat/absolute.h>
#include <passport/infra/libs/cpp/unistat/diff.h>
#include <passport/infra/libs/cpp/unistat/time_stat.h>

#include <util/generic/string.h>

#include <map>
#include <vector>

class TThreadPool;

namespace NPassport::NUnistat {
    class TBuilder;
}

namespace NPassport::NKolmogor {
    struct TSpaceSettings {
        TString Name;
        size_t SliceCount;
        size_t Reserve;
        ui32 CounterTtl;
        ui32 CounterCount;
        ui32 ThreadCount = 1;
        size_t EraseCount = 1024;
        bool Persistency = true;
        std::optional<size_t> MemoryLimit;
        TSmallVec<ui32> AllowedClientIds = {};
    };

    struct TSpaceUnistat {
        TSpaceUnistat(const TString& name, std::optional<size_t> memoryLimit);
        TSpaceUnistat(int forTotal);

        NUnistat::TSignalDiff<> GetRequests;
        NUnistat::TSignalDiff<> GetKeys;
        NUnistat::TSignalDiff<> IncRequests;
        NUnistat::TSignalDiff<> IncKeys;
        NUnistat::TSignalDiff<> EraseRequests;
        NUnistat::TSignalDiff<> EraseKeys;

        TSpaceSizeStatsPtr Space;

    private:
        TSpaceUnistat(int, const TString& id, std::optional<size_t> memoryLimit = {});
    };

    class TSpace {
    public:
        TSpace(const TSpaceSettings& conf, const TString& dataDirectory, size_t cleanCount);
        ~TSpace();

        TSpace(const TSpace&) = delete;
        TSpace& operator=(const TSpace&) = delete;
        TSpace(TSpace&&) = delete;
        TSpace& operator=(TSpace&&) = delete;

        enum class EAPI {
            V1,
            V2,
        };

        NV2::TGetResult::TValues IncAndGet(const NV2::TIncRequest::TSpace& keys, TInstant now); // for http service
        void Inc(const TStrVec& keys, TInstant instant);                                        // for replicator
        NV2::TGetResult::TValues Get(const TStrVec& keys, EAPI apiVer) const;
        TInstant Erase(const TString& rawKeys);            // for http service
        void Erase(const TStrVec& keys, TInstant instant); // for replicator
        ui32 EraseAll();                                   // for http service

        void Clean();

        struct TStats {
            ui64 GetRequests = 0;
            ui64 GetKeys = 0;
            ui64 IncRequests = 0;
            ui64 IncKeys = 0;
            ui64 EraseRequests = 0;
            ui64 EraseKeys = 0;

            size_t Keys = 0;
            size_t Memory = 0;

            void operator+=(const TStats& o);
        };
        TStats AddUnistat(NUnistat::TBuilder& builder);

        const TCounterParams& Params() const;

        void Load(TThreadPool& queue);
        void Save(TThreadPool& queue) const;

    private:
        void EraseImpl(const TStrVec& keys, TInstant ins);
        using TIdxs = std::map<size_t, TIdxVec>;
        template <class T>
        TIdxs GetIdxs(const T& keys) const;

        size_t GetBucketIdx(const TString& s) const;

        void EnsurePath() const;
        void ClearPath() const;
        bool IsConfigConsistent() const;
        void ValidateConfigConsistency() const;
        void SaveConfig() const;

        std::vector<TSlice> Arr_;
        const TCounterParams Params_;
        const TString Name_;
        TString DataDirectory_;
        TString SpaceDataDirectory_;
        bool Persistency_;
        TKolmoPeriod LastCleanedPeriod_;

        mutable TSpaceUnistat Unistat_;
        NUnistat::TTimeStat UnistatCleanDuration_;
    };

    inline const TCounterParams& TSpace::Params() const {
        return Params_;
    }
}
