#include <passport/infra/daemons/kolmogor/src/common/auth.h>
#include <passport/infra/daemons/kolmogor/src/common/exception.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/tvmauth/client/facade.h>
#include <library/cpp/tvmauth/client/mocked_updater.h>

using namespace NPassport::NKolmogor;

Y_UNIT_TEST_SUITE(Auth) {
    const TString REPL_TICKET = "3:serv:CBAQ__________9_IgYI8gEQ8gE:Ck9Bz0Pi2pQtJhM9_Lvlpqm3ZHBXDRHefJqTVKucGekbBm9hANu5Kz_AkBweCRuYTN2g9aX6XCKo1VytsEsypYnKtZTRjtcWMVdvTYHjzQY7IU-H3M6Nzro_QCQcLvWCO_rOQPXwa39xrv92A_baLElvaALQB1FLlUnO_-yCoLE";
    const TString CONSUMER_TICKET = "3:serv:CBAQ__________9_IgcIlZEGEPIB:GT9N3CKoxJAMZD7aEeUpmzc8YppwQ88pcMen7BsaGf-r5G36kaj9tc7nKQM23kmwGT9sUgr1f0r1pYGhXXRH3TOIaeOPB6HSNCNnj3TlqaCIlfGKWTfmcZNUTIEAKFiOJJV6YFiZp5TvU7JAqemwlzpa1M869QYudhE4XdVDQuU";

    class TAuthTest: public TAuth {
    public:
        using TAuth::Acl_;
        using TAuth::TAcl;
        using TAuth::TAuth;
    };

    static TAuthTest CreateAuth() {
        auto c = std::make_unique<NTvmAuth::TTvmClient>(
            MakeIntrusiveConst<NTvmAuth::TMockedUpdater>(
                NTvmAuth::TMockedUpdater::TSettings{
                    .SelfTvmId = 242,
                    .Backends = {
                        {.Alias = "kolmogor", .Id = 242, .Value = REPL_TICKET},
                    },
                }));

        return TAuthTest(std::move(c), {242});
    }

    Y_UNIT_TEST(Create) {
        TAuthTest auth = CreateAuth();
        UNIT_ASSERT(auth.Acl_.empty());

        auth.AddAcl("test", {});
        UNIT_ASSERT(auth.Acl_.empty());

        auth.AddAcl("test", {1, 3, 2});
        UNIT_ASSERT_EQUAL(TAuthTest::TAcl({{"test", {1, 2, 3}}}), auth.Acl_);

        auth.AddAcl("test", {4, 5, 1});
        UNIT_ASSERT_EQUAL(TAuthTest::TAcl({{"test", {1, 2, 3}}}), auth.Acl_);
    }

    Y_UNIT_TEST(Replication) {
        TAuthTest auth = CreateAuth();
        UNIT_ASSERT_STRINGS_EQUAL(REPL_TICKET, auth.GetServiceTicketForReplication());
        UNIT_ASSERT_NO_EXCEPTION(auth.CheckServiceTicketForReplication(REPL_TICKET));
        UNIT_ASSERT_EXCEPTION_CONTAINS(auth.CheckServiceTicketForReplication(CONSUMER_TICKET),
                                       TAuthException,
                                       "Service ticket is not allowed for 'replication': 100501");
    }

    Y_UNIT_TEST(Check) {
        TAuthTest auth = CreateAuth();
        UNIT_ASSERT_NO_EXCEPTION(auth.CheckServiceTicket({}, {"test"}));
        UNIT_ASSERT_NO_EXCEPTION(auth.CheckServiceTicket({}, {"foo"}));

        auth.AddAcl("test", {100501, 3, 2});
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            auth.CheckServiceTicket({}, {"test"}),
            TAuthException,
            "Service ticket required but not found for 'test'");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            auth.CheckServiceTicket(123, {"test"}),
            TAuthException,
            "Service ticket is not allowed for 'test'. Your tvm_id: 123");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            auth.CheckServiceTicket(242, {"test"}),
            TAuthException,
            "Service ticket is not allowed for 'test'. Your tvm_id: 242");
        UNIT_ASSERT_NO_EXCEPTION(auth.CheckServiceTicket(100501, {"test"}));
        UNIT_ASSERT_NO_EXCEPTION(auth.CheckServiceTicket({}, {"foo"}));
    }
}
