#include <passport/infra/daemons/kolmogor/src/common/exception.h>
#include <passport/infra/daemons/kolmogor/src/storage/mem_storage.h>
#include <passport/infra/daemons/kolmogor/src/storage/space.h>

#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <library/cpp/config/config.h>
#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <util/stream/file.h>
#include <util/string/cast.h>
#include <util/string/split.h>

#include <ctime>

using namespace NPassport::NKolmogor;

Y_UNIT_TEST_SUITE(MemStorageTest) {
    static NV2::TGetResult CreateResult(ui32 abc, ui32 asd, ui32 qwe, ui32 rty) {
        return NV2::TGetResult{
            .Data = {
                NV2::TGetResult::TSpace{
                    .Name = "test2",
                    .Values = {
                        {.Key = "abc", .Value = abc},
                        {.Key = "asd", .Value = asd},
                        {.Key = "qwe", .Value = qwe},
                        {.Key = "rty", .Value = rty},
                    },
                },
            },
        };
    }

    Y_UNIT_TEST(MemStorageTest) {
        ui32 now = time(nullptr);

        const NV2::TGetRequest req = {
            .Req = {
                {
                    .Name = "test2",
                    .Keys = {"abc", "qwe", "asd", "rty"},
                },
            },
        };

        TMemStorage m;
        UNIT_ASSERT_EXCEPTION(m.Get(req, TSpace::EAPI::V2), TBadRequestException);
        UNIT_ASSERT(!m.HasSpace("test2"));

        m.AddSpace({"test2", 4, 4, 1000, 100, 1, 2048, false, {}});
        UNIT_ASSERT(m.HasSpace("test2"));
        UNIT_ASSERT_VALUES_EQUAL(CreateResult(0, 0, 0, 0), m.Get(req, TSpace::EAPI::V2));

        m.Inc("test2", TStrVec({"qwe", "abc"}), TInstant::Seconds(now));
        UNIT_ASSERT_VALUES_EQUAL(CreateResult(1, 0, 1, 0), m.Get(req, TSpace::EAPI::V2));

        m.Inc("test2", TStrVec({"asd"}), TInstant::Seconds(now));
        m.Inc("test2", TStrVec({"asd"}), TInstant::Seconds(now));
        m.Inc("test2", TStrVec({"asd"}), TInstant::Seconds(now));
        UNIT_ASSERT_VALUES_EQUAL(CreateResult(1, 3, 1, 0), m.Get(req, TSpace::EAPI::V2));

        m.Erase("test2", TStrVec({"abc", "qwe"}), TInstant::Seconds(now));
        UNIT_ASSERT_VALUES_EQUAL(CreateResult(0, 3, 0, 0), m.Get(req, TSpace::EAPI::V2));

        UNIT_ASSERT_EXCEPTION_CONTAINS(m.Erase("tes", ""),
                                       yexception,
                                       "Space is not found (erase): tes");
    }

    Y_UNIT_TEST(PersistencyTest) {
        ui32 now = time(nullptr);
        const int supposedCounterCount = 12; // На 1 меньше реального количества, чтобы защитится от флапа
        const int supposedActualLength = 300;

        const TString dataPath = "./testing_out_stuff/data";
        auto create = [dataPath]() {
            return TMemStorage(TMemStorageSettings{.DataDirectory = dataPath});
        };

        TMemStorage m1 = create();
        TMemStorage m2 = create();
        TMemStorage m3 = create();

        {
            TSpaceSettings spaceConfig{
                .Name = "test",
                .SliceCount = 4,
                .Reserve = 2097152,
                .CounterTtl = 3900,
                .CounterCount = 13,
                .ThreadCount = 1,
                .Persistency = true,
            };

            m1.AddSpace(spaceConfig);
            m2.AddSpace(spaceConfig);
            // Создать MemStorage m3 для проверки того, что счетчики не загрузятся при измененном конфиге
            spaceConfig.SliceCount += 1;
            m3.AddSpace(spaceConfig);
        }

        for (int i = 0; i < 120; ++i) {
            for (int j = 0; j < 10; ++j) {
                m1.Inc(
                    "test",
                    TStrVec({"abc" + IntToString<10>(j)}),
                    TInstant::Seconds(now - supposedActualLength * (i % supposedCounterCount)));
            }
        }

        auto createReq = [](const TString& key) {
            return NV2::TGetRequest{
                .Req = {
                    {
                        .Name = "test",
                        .Keys = {key},
                    },
                },
            };
        };
        auto createRes = [](const TString& key, ui64 value) {
            return NV2::TGetResult{
                .Data = {
                    NV2::TGetResult::TSpace{
                        .Name = "test",
                        .Values = {
                            {.Key = key, .Value = value},
                        },
                    },
                },
            };
        };

        m1.SaveOnExit();
        m2.LoadOnStart();
        for (int i = 0; i < 10; ++i) {
            const TString ix = "abc" + IntToString<10>(i);
            NV2::TGetRequest req = createReq(ix);
            UNIT_ASSERT_VALUES_EQUAL(m1.Get(req, TSpace::EAPI::V2), createRes(ix, 120));
            UNIT_ASSERT_VALUES_EQUAL(m1.Get(req, TSpace::EAPI::V2), m2.Get(req, TSpace::EAPI::V2));
        }
        TFileInput currentDump(dataPath + "/test/metadata");
        UNIT_ASSERT_VALUES_EQUAL("4", currentDump.ReadAll());

        m3.LoadOnStart();
        for (int i = 0; i < 10; ++i) {
            const TString ix = "abc" + IntToString<10>(i);
            UNIT_ASSERT_VALUES_EQUAL(m3.Get(createReq(ix), TSpace::EAPI::V2), createRes(ix, 0));
        }
    }
}

template <>
void Out<NV2::TGetResult>(IOutputStream& o, const NV2::TGetResult& value) {
    for (const NV2::TGetResult::TSpace& space : value.Data) {
        o << space << Endl;
    }
}

template <>
void Out<NV2::TGetResult::TSpace>(IOutputStream& o, const NV2::TGetResult::TSpace& value) {
    o << value.Name << ". " << value.Values << "...";
}

template <>
void Out<NV2::TGetResult::TValues>(IOutputStream& o, const NV2::TGetResult::TValues& value) {
    for (const auto& v : value) {
        o << v << ";";
    }
}

template <>
void Out<NV2::TGetResult::TKeyValue>(IOutputStream& o, const NV2::TGetResult::TKeyValue& value) {
    o << value.Key << "->" << value.Value;
}
