#include <passport/infra/daemons/kolmogor/src/storage/slice.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NKolmogor;

Y_UNIT_TEST_SUITE(SliceTest) {
    static const NV2::TGetResult::TValues ZERO(
        {
            {.Key = "abc", .Value = 0},
            {.Key = "qwe", .Value = 0},
            {.Key = "asd", .Value = 0},
            {.Key = "rty", .Value = 0},
        });

    Y_UNIT_TEST(TooOldData) {
        TCounterParams params(1000, 100);

        TSlice s(100, params, 1024, 2048, std::make_shared<TSpaceSizeStats>(""));

        TKolmoPeriod now = {1200};
        TStrVec keys({"abc", "qwe", "asd", "rty"});
        TIdxVec idxs({0, 1, 2, 3});

        UNIT_ASSERT(s.Inc(keys, idxs, TInstant::Seconds(100), now));

        NV2::TGetResult::TValues out = ZERO;
        s.Get(TIdxVec({0, 1, 2, 3}), now, out);

        UNIT_ASSERT_VALUES_EQUAL(ZERO, out);
    }

    Y_UNIT_TEST(OkData) {
        TCounterParams params(1000, 100);

        TSlice s(100, params, 1024, 2048, std::make_shared<TSpaceSizeStats>(""));

        TInstant nowIns = TInstant::Seconds(12000);
        TKolmoPeriod now = params.FromInstant(nowIns);
        TStrVec keys({"abc", "qwe", "asd", "rty"});
        TIdxVec idxs({0, 2, 3});

        s.Inc(keys, idxs, nowIns, now);

        NV2::TGetResult::TValues out = ZERO;
        s.Get(TIdxVec({0, 2, 3}), now, out);

        UNIT_ASSERT_VALUES_EQUAL(NV2::TGetResult::TValues(
                                     {
                                         {.Key = "abc", .Value = 1},
                                         {.Key = "qwe", .Value = 0},
                                         {.Key = "asd", .Value = 1},
                                         {.Key = "rty", .Value = 1},
                                     }),
                                 out);

        s.Erase({"abc", "qwe", "asd"}, {0, 1, 2}, nowIns);

        out = ZERO;
        s.Get(TIdxVec({0, 2, 3}), now, out);

        UNIT_ASSERT_VALUES_EQUAL(NV2::TGetResult::TValues(
                                     {
                                         {.Key = "abc", .Value = 0},
                                         {.Key = "qwe", .Value = 0},
                                         {.Key = "asd", .Value = 0},
                                         {.Key = "rty", .Value = 1},
                                     }),
                                 out);
    }

    Y_UNIT_TEST(OldData) {
        TCounterParams params(1000, 100);

        TSlice s(100, params, 1024, 2048, std::make_shared<TSpaceSizeStats>(""));

        TKolmoPeriod now = {1200};
        TStrVec keys({"abc", "qwe", "asd", "rty"});
        TIdxVec idxs({0, 2, 3});

        UNIT_ASSERT(s.Inc(keys, idxs, params.ToInstantBegining(now), now));
        UNIT_ASSERT(s.Inc(keys, idxs, params.ToInstantBegining({now.Val - 50}), now));
        UNIT_ASSERT(s.Inc(keys, idxs, TInstant::Seconds(100), now));

        NV2::TGetResult::TValues out = ZERO;
        s.Get(TIdxVec({0, 2, 3}), now, out);

        UNIT_ASSERT_VALUES_EQUAL(NV2::TGetResult::TValues(
                                     {
                                         {.Key = "abc", .Value = 2},
                                         {.Key = "qwe", .Value = 0},
                                         {.Key = "asd", .Value = 2},
                                         {.Key = "rty", .Value = 2},
                                     }),
                                 out);

        s.Erase({"abc", "qwe", "asd", "lsjkbsf"}, {0, 1, 2, 3}, params.ToInstantBegining({now.Val - 25}));

        out = ZERO;
        s.Get(TIdxVec({0, 2, 3}), now, out);

        UNIT_ASSERT_VALUES_EQUAL(NV2::TGetResult::TValues(
                                     {
                                         {.Key = "abc", .Value = 1},
                                         {.Key = "qwe", .Value = 0},
                                         {.Key = "asd", .Value = 1},
                                         {.Key = "rty", .Value = 2},
                                     }),
                                 out);
    }

    Y_UNIT_TEST(Clean) {
        TCounterParams params(1000, 100);

        TSlice s(100, params, 50, 50, std::make_shared<TSpaceSizeStats>(""));

        ui32 now = 1200;
        TStrVec keys({"abc", "qwe", "asd", "rty"});
        TIdxVec idxs({0, 2, 3});

        UNIT_ASSERT(s.Inc(keys, idxs, TInstant::Seconds(now), {now}));
        UNIT_ASSERT(s.Inc(keys, idxs, TInstant::Seconds(now - 50), {now}));
        UNIT_ASSERT(s.Inc(keys, idxs, TInstant::Seconds(100), {now}));

        TSlice::TCleanInfo info = s.Clean({now + 1000}, 0);
        UNIT_ASSERT(info.IsDirty);
        UNIT_ASSERT_VALUES_EQUAL(50, info.LastIdx);
        UNIT_ASSERT_VALUES_EQUAL(2, info.Deleted);

        info = s.Clean({now + 1000}, 50);
        UNIT_ASSERT(info.IsDirty);
        UNIT_ASSERT_VALUES_EQUAL(100, info.LastIdx);
        UNIT_ASSERT_VALUES_EQUAL(1, info.Deleted);

        info = s.Clean({now + 1000}, 100);
        UNIT_ASSERT(info.IsDirty);
        UNIT_ASSERT_VALUES_EQUAL(150, info.LastIdx);
        UNIT_ASSERT_VALUES_EQUAL(0, info.Deleted);

        info = s.Clean({now + 1000}, 150);
        UNIT_ASSERT(info.IsDirty);
        UNIT_ASSERT_VALUES_EQUAL(200, info.LastIdx);
        UNIT_ASSERT_VALUES_EQUAL(0, info.Deleted);
    }

    Y_UNIT_TEST(EraseAll) {
        TCounterParams params(1000, 100);

        TSlice s(100, params, 50, 50, std::make_shared<TSpaceSizeStats>(""));

        ui32 now = 2000;
        TInstant nowInst = TInstant::Seconds(now);

        TStrVec keys({"abc", "qwe", "asd", "rty"});
        TIdxVec idxs({0, 2, 3});

        UNIT_ASSERT(s.Inc(keys, idxs, nowInst, params.FromInstant(nowInst)));
        UNIT_ASSERT(s.Inc(keys, idxs, nowInst, params.FromInstant(nowInst)));

        NV2::TGetResult::TValues out = ZERO;
        s.Get(idxs, params.FromInstant(nowInst), out);

        UNIT_ASSERT_VALUES_EQUAL(NV2::TGetResult::TValues(
                                     {
                                         {.Key = "abc", .Value = 2},
                                         {.Key = "qwe", .Value = 0},
                                         {.Key = "asd", .Value = 2},
                                         {.Key = "rty", .Value = 2},
                                     }),
                                 out);

        TSlice::TCleanInfo info = s.EraseAll(TInstant::Seconds(now + 100), 0);
        UNIT_ASSERT(info.IsDirty);
        UNIT_ASSERT_VALUES_EQUAL(info.Deleted, 2);
        UNIT_ASSERT_VALUES_EQUAL(info.LastIdx, 50);

        out = ZERO;
        s.Get(idxs, params.FromInstant(nowInst), out);

        UNIT_ASSERT_VALUES_EQUAL(NV2::TGetResult::TValues(
                                     {
                                         {.Key = "abc", .Value = 2},
                                         {.Key = "qwe", .Value = 0},
                                         {.Key = "asd", .Value = 0},
                                         {.Key = "rty", .Value = 0},
                                     }),
                                 out);

        info = s.EraseAll(TInstant::Seconds(now + 100), 50);
        UNIT_ASSERT(info.IsDirty);
        UNIT_ASSERT_VALUES_EQUAL(info.Deleted, 1);
        UNIT_ASSERT_VALUES_EQUAL(info.LastIdx, 100);

        out = ZERO;
        s.Get(idxs, params.FromInstant(nowInst), out);

        UNIT_ASSERT_VALUES_EQUAL(NV2::TGetResult::TValues(
                                     {
                                         {.Key = "abc", .Value = 0},
                                         {.Key = "qwe", .Value = 0},
                                         {.Key = "asd", .Value = 0},
                                         {.Key = "rty", .Value = 0},
                                     }),
                                 out);

        info = s.EraseAll(TInstant::Seconds(now + 100), 100);
        UNIT_ASSERT(info.IsDirty);
        UNIT_ASSERT_VALUES_EQUAL(info.Deleted, 0);
        UNIT_ASSERT_VALUES_EQUAL(info.LastIdx, 150);

        info = s.EraseAll(TInstant::Seconds(now + 100), 150);
        UNIT_ASSERT(info.IsDirty);
        UNIT_ASSERT_VALUES_EQUAL(info.Deleted, 0);
        UNIT_ASSERT_VALUES_EQUAL(info.LastIdx, 200);

        info = s.EraseAll(TInstant::Seconds(now + 100), 200);
        UNIT_ASSERT(!info.IsDirty);
        UNIT_ASSERT_VALUES_EQUAL(info.Deleted, 0);
        UNIT_ASSERT_VALUES_EQUAL(info.LastIdx, 211);
    }

    Y_UNIT_TEST(Stats) {
        TCounterParams params(1000, 100);
        TSpaceSizeStatsPtr stats = std::make_shared<TSpaceSizeStats>("");

        TSlice s(100, params, 1024, 2048, stats);

        ui32 now = 1200;
        TStrVec keys({"abc", "qwe", "asd", "rty"});
        TIdxVec idxs({0, 2, 3});

        UNIT_ASSERT(s.Inc(keys, idxs, TInstant::Seconds(now), {now}));
        UNIT_ASSERT(s.Inc(keys, idxs, TInstant::Seconds(now - 50), {now}));
        UNIT_ASSERT(s.Inc(keys, idxs, TInstant::Seconds(100), {now}));

        UNIT_ASSERT_VALUES_EQUAL(3, stats->Keys().GetValue());
        UNIT_ASSERT_VALUES_EQUAL(2553, stats->Memory().GetValue());

        s.Erase({"asd", "vbn", "xc"}, {0, 1, 2}, TInstant::Seconds(now));
        UNIT_ASSERT_VALUES_EQUAL(5, stats->Keys().GetValue());
        UNIT_ASSERT_VALUES_EQUAL(4254, stats->Memory().GetValue());
    }

    Y_UNIT_TEST(Limited) {
        TCounterParams params(1000, 100);
        TSpaceSizeStatsPtr stats = std::make_shared<TSpaceSizeStats>("", 2048);

        TSlice s(100, params, 1024, 2048, stats);

        ui32 now = 1200;
        TStrVec keys({"abc", "qwe", "asd"});
        TIdxVec idxs({0, 1, 2});

        UNIT_ASSERT(!s.Inc(keys, idxs, TInstant::Seconds(now), {now}));

        idxs.pop_back();
        UNIT_ASSERT(s.Inc(keys, idxs, TInstant::Seconds(now), {now}));
        UNIT_ASSERT(s.Inc(keys, idxs, TInstant::Seconds(now), {now}));

        idxs.push_back(2);
        UNIT_ASSERT(!s.Inc(keys, idxs, TInstant::Seconds(now), {now}));

        UNIT_ASSERT_VALUES_EQUAL(2, stats->Keys().GetValue());
        UNIT_ASSERT_VALUES_EQUAL(1702, stats->Memory().GetValue());

        UNIT_ASSERT(!s.Erase({"abc", "vbn"}, {0, 1}, TInstant::Seconds(now)));
        UNIT_ASSERT(s.Erase({"abc"}, {0}, TInstant::Seconds(now)));
    }
}
