#include "data_pusher.h"

#include "yt_schema.h"

#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/exponential_backoff.h>
#include <passport/infra/libs/cpp/utils/log/global.h>

#include <util/random/random.h>
#include <util/string/join.h>
#include <util/system/thread.h>

#include <atomic>
#include <thread>

namespace NPassport::NLbcbck {
    TDataPusher::TDataPusher(TLbReaderPoolPtr readerPool)
        : TBase(NLb::NPusher::TBaseSettings{
                    .SignalPrefix = "yt",
                    .ErrorSignalName = "yt.errors",
                }, readerPool)
    {
        Unistat.TimeStats = std::make_shared<NUnistat::TTimeStat>(
            "yt.response_time",
            NUnistat::TTimeStat::CreateBoundsFromMaxValue(TDuration::Seconds(5)));
    }

    TDataPusher::~TDataPusher() {
        Stop();
    }

    void TDataPusher::AddUnistat(NUnistat::TBuilder& builder) const {
        AddUnistatBase(builder);

        builder.Add(Unistat.Requests);
        builder.Add(Unistat.WrittenBytes);
        Unistat.TimeStats->AddUnistat(builder);
    }

    TDataPushWorker::TDataPushWorker(TSettings settings, size_t idx, TDataPusher& parent)
        : TBase(settings.Base, idx, parent, [this](auto& d) { return Proc(d); })
        , BatchSettings_(settings.BatchSettings)
        , Client_(settings.Yt->Create())
    {
    }

    bool TDataPushWorker::Proc(TLbReaderPool::TParsedData& data) {
        const TString cookies = JoinSeq(",", data.Traceids);

        NYt::TBatch batch(BatchSettings_, Client_);

        for (auto& [table, builder] : data.Data.ByTable) {
            batch.Send(
                table.TableName(),
                *builder,
                [&](NYt::TBatch::TQueryResult& req) {
                    ++GetParent().Unistat.Requests;
                    GetParent().Unistat.WrittenBytes += req.Size;

                    req.Future.Subscribe(
                        [timeStat = GetParent().Unistat.TimeStats, start = TInstant::Now()](const auto&) {
                            timeStat->Insert(TInstant::Now() - start);
                        });
                });
        }

        const bool result = batch.WaitResult(
            [&](NYt::TBatch::TQueryResult& req) {
                TLog::Debug() << "worker#" << Idx_
                              << ". Trying to write " << req.Rows << " rows to " << req.TableName
                              << ". Result: " << req.Future.GetValue().Msg
                              << ". Cookies: " << cookies
                              << ". Request time: " << TInstant::Now() - req.StartTime;
                return req.Future.GetValue().Status == NYt::TYtClient::TWriteResult::Ok;
            });
        if (!result) {
            return false;
        }

        for (NThreading::TPromise<void>& p : data.Promises) {
            p.SetValue();
        }
        return true;
    }
}
