#include "lb_parser.h"

#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/split.h>

#include <util/generic/hash.h>

namespace NPassport::NLbcbck {
    static const THashMap<TStringBuf, TLbParser::TRowCtx::EKey> TOKENS = {
        {"tskv_format", TLbParser::TRowCtx::EKey::Format},
        {"action", TLbParser::TRowCtx::EKey::Action},
        {"status", TLbParser::TRowCtx::EKey::Status},
        {"userip", TLbParser::TRowCtx::EKey::UserIp},
        {"user_ip", TLbParser::TRowCtx::EKey::UserIp},
        {"unixtime", TLbParser::TRowCtx::EKey::Unixtime},
        {"host", TLbParser::TRowCtx::EKey::Host},
        {"def_uid", TLbParser::TRowCtx::EKey::DefUid},
        {"uids", TLbParser::TRowCtx::EKey::Uids},
        {"uid", TLbParser::TRowCtx::EKey::Uid},
        {"user_port", TLbParser::TRowCtx::EKey::UserPort},
        {"client_id", TLbParser::TRowCtx::EKey::ClientId},
    };

    TLbParser::TRowCtx::TRowCtx(TStringBuf buf) {
        Y_ENSURE(buf.SkipPrefix("tskv\t"), "string is not tskv");
        while (buf) {
            TStringBuf pair = buf.NextTok('\t');
            auto it = TOKENS.find(pair.NextTok('='));
            if (it != TOKENS.end()) {
                Values_[(size_t)it->second] = pair;
            }
        }
    }

    TStringBuf TLbParser::TRowCtx::Get(TLbParser::TRowCtx::EKey key) const {
        size_t idx = (size_t)key;
        Y_VERIFY(idx < (size_t)TLbParser::TRowCtx::EKey::Count);
        return Values_[idx];
    }

    static const TString BLACKBOX_STATBOX_LOG = "blackbox_statbox_log";

    TLbParser::TLbParser()
        : NLb::TBaseParser(BLACKBOX_STATBOX_LOG)
    {
    }

    TLbParser::TResult TLbParser::Parse(const NLb::TDataSet<>& data) {
        ParseRows(data);
        return std::move(Result_);
    }

    bool TLbParser::ParseRow(TStringBuf line) {
        std::optional<TWritableRow> r = TryParse(line);
        if (r) {
            if (UniqueRows_.insert(*r).second) {
                ++TotalRowsOut_;
                GetConverter(Result_, *r, Reserve_).Add(std::move(*r));
            }
            return true;
        }

        return false;
    }

    void TLbParser::Reserve(size_t size) {
        Reserve_ = size;
    }

    void TLbParser::LogBadLine(TStringBuf) {
        // TODO
    }

    TRowsConverter& TLbParser::GetConverter(TResult& result,
                                            const TWritableRow& row,
                                            size_t reserve) {
        auto it = result.ByTable.find(row.Unixtime);
        if (it != result.ByTable.end()) {
            return static_cast<TRowsConverter&>(*it->second);
        }

        it = result.ByTable.emplace(
                               NYt::TDailyTable(row.Unixtime),
                               std::make_unique<TRowsConverter>())
                 .first;
        TRowsConverter& res = static_cast<TRowsConverter&>(*it->second);

        res.Reserve(reserve);
        return res;
    }

    std::optional<TWritableRow> TLbParser::TryParse(TStringBuf line) {
        TRowCtx ctx(line);

        if (ctx.Get(TRowCtx::EKey::Format) == "blackbox-log" && ctx.Get(TRowCtx::EKey::Action) == "sescheck") {
            return ParseAsSescheck(ctx);
        }

        if (ctx.Get(TRowCtx::EKey::Format) == "oauth-log" && ctx.Get(TRowCtx::EKey::Status) == "OK") {
            return ParseAsOauthcheck(ctx);
        }

        return {};
    }

    TWritableRow TLbParser::ParseAsSescheck(const TRowCtx& ctx) {
        TWritableRow res;
        res.UserIp = TString(ctx.Get(TRowCtx::EKey::UserIp));
        res.HostClient = TString(ctx.Get(TRowCtx::EKey::Host));
        Y_ENSURE(TryIntFromString<10>(ctx.Get(TRowCtx::EKey::Unixtime), res.Unixtime),
                 "'unixtime' must be int");

        TStringBuf defUidStr = ctx.Get(TRowCtx::EKey::DefUid);
        if (defUidStr) {
            Y_ENSURE(TryIntFromString<10>(defUidStr, res.DefUid),
                     "'def_uid' must be int");
        }

        TStringBuf uidsStr = ctx.Get(TRowCtx::EKey::Uids);
        while (uidsStr) {
            TStringBuf uid = uidsStr.NextTok(',');
            if (!uid || defUidStr == uid) {
                continue;
            }

            if (res.Uids.empty()) {
                res.Uids.reserve(64);
            } else {
                res.Uids.push_back(',');
            }
            res.Uids.append(uid);
        }

        TStringBuf userPort = ctx.Get(TRowCtx::EKey::UserPort);
        if (userPort && !TryIntFromString<10>(userPort, res.UserPort)) {
            TLog::Warning() << "parseAsSescheck: invalid user_port: '" << userPort << "'";
        }

        return res;
    }

    std::optional<TWritableRow> TLbParser::ParseAsOauthcheck(const TRowCtx& ctx) {
        TStringBuf uid = ctx.Get(TRowCtx::EKey::Uid);
        if (uid.empty()) {
            return {};
        }

        TWritableRow res;
        res.UserIp = TString(ctx.Get(TRowCtx::EKey::UserIp));
        res.HostClient = TString(ctx.Get(TRowCtx::EKey::ClientId));
        Y_ENSURE(TryIntFromString<10>(ctx.Get(TRowCtx::EKey::Unixtime), res.Unixtime),
                 "'unixtime' must be int");
        Y_ENSURE(TryIntFromString<10>(uid, res.DefUid),
                 "'uid' must be int");

        TStringBuf userPort = ctx.Get(TRowCtx::EKey::UserPort);
        if (userPort && !TryIntFromString<10>(userPort, res.UserPort)) {
            TLog::Warning() << "parseAsOauthcheck: invalid user_port: '" << userPort << "'";
        }

        return res;
    }
}
