#include "lbcbck.h"

#include "lb_parser.h"
#include "yt_schema.h"

#include <passport/infra/libs/cpp/json/reader.h>
#include <passport/infra/libs/cpp/tvm/logger/logger.h>
#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/xml/config.h>

#include <library/cpp/tvmauth/client/misc/tool/settings.h>

#include <util/stream/file.h>
#include <util/system/env.h>

namespace NPassport::NLbcbck {
    static const TString TABLE = "kek";

    TLbcbck::TLbcbck() = default;

    TLbcbck::~TLbcbck() {
        Stop();
        TLog::Debug() << "TLbcbck was destroyed";
    }

    void TLbcbck::Init(const NXml::TConfig& config) {
        const TString path = "/config/component";

        config.InitCommonLog(path + "/logger");

        InitTvmtool(config, path + "/tvmtool");
        NYt::TBatchSettings settings = InitYt(config, path + "/yt");
        InitLogbroker(config, path + "/logbroker");
        InitProcessor(config, path + "/processor", settings);

        Start();
    }

    void TLbcbck::HandleRequest(NCommon::TRequest& req) {
        TStringBuf path = req.GetPath();
        path.ChopSuffix("/");

        if (path != "/ping") {
            TLog::Warning() << "Unknown path: " << path;
            req.SetStatus(HTTP_NOT_FOUND);
            req.Write(TStringBuilder() << "Unknown path: " << path);
            return;
        }

        TStringStream s;
        s << Lb_->GetErrorInfo();

        NTvmAuth::TClientStatus tvmStatus = Tvm_->GetStatus();
        if (!(tvmStatus == NTvmAuth::TClientStatus::Ok)) {
            s << "TvmClient is bad: " << tvmStatus.GetLastError() << Endl;
        }

        if (s.empty()) {
            req.SetStatus(HTTP_OK);
            req.Write("OK\n");
        } else {
            s << Endl;
            req.SetStatus(HTTP_INTERNAL_SERVER_ERROR);
            req.Write(s.Str());
        }
    }

    void TLbcbck::AddUnistat(NUnistat::TBuilder& builder) {
        DataPusher_->AddUnistat(builder);
        Lb_->AddUnistat(builder);
        UncommittedMemoryDispatcher_->AddUnistat(builder);
        builder.Add(UnistatRowsIn_);
        builder.Add(UnistatRowsOut_);
        builder.Add(UnistatErrors_);
    }

    void TLbcbck::RotateLogs() {
        YtFactory_->RotateLogs();
    }

    void TLbcbck::InitTvmtool(const NXml::TConfig& config, const TString& path) {
        NTvmAuth::NTvmTool::TClientSettings setts("lbcbck");
        setts.SetPort(config.AsInt(path + "/port"));
        setts.SetAuthToken(NUtils::ReadFile(config.AsString(path + "/authtoken")));

        Tvm_ = std::make_shared<NTvmAuth::TTvmClient>(setts, NTvmLogger::TLogger::Create());
    }

    void TLbcbck::InitLogbroker(const NXml::TConfig& config, const TString& path) {
        Lb_ = std::make_shared<TLbReaderPool>(
            NLb::TReaderPoolSettings{
                .QueueSizeLimit = config.AsNum<ui32>(path + "/queue_size"),
                .Workers = config.AsNum<ui32>(path + "/queue_workers"),
            });

        NLb::TReaderSettings setts;
        setts.LogLevel = config.AsInt(path + "/log_level");
        setts.ClientId = config.AsString(path + "/client_id");
        setts.MaxMemoryUsage = 1024 * 1024 * config.AsNum<size_t>(path + "/memory_usage_per_server__mb");
        setts.ReadMaxBytes = 1024 * 1024 * config.AsNum<size_t>(path + "/read_max_bytes__mb");
        setts.ReadMaxCounts = config.AsNum<size_t>(path + "/read_max_count", setts.ReadMaxCounts);
        setts.MaxUncommittedSize = config.AsNum<size_t>(path + "/max_uncommitted_size", setts.MaxUncommittedSize);

        if (config.Contains(path + "/port")) { // For tests
            setts.ServerPort = config.As<ui16>(path + "/port");
        } else {
            setts.TvmClient = Tvm_;
        }

        UncommittedMemoryDispatcher_ = setts.UncommittedMemory;

        for (const TString& topicAlias : {"blackbox-log", "oauth-log"}) {
            TString topic = config.AsString(path + "/topics/" + topicAlias);
            setts.Topics = {topic};
            for (const TString& p : config.SubKeys(path + "/servers/server")) {
                setts.ServerName = config.AsString(p);
                setts.UnistatSignalId = NUtils::CreateStr(setts.ServerName, "/", topic);
                Lb_->AddReader(setts);
            }
        }
        Y_ENSURE(Lb_->GetReadersCount(), "There is no servers or topics in config");
    }

    NYt::TBatchSettings TLbcbck::InitYt(const NXml::TConfig& config, const TString& path) {
        SetEnv("TZ", "Europe/Moscow");
        tzset();

        NYt::TYtSettings settings;

        {
            TFileInput file(config.AsString(path + "/oauth_token_file"));
            TString body = file.ReadAll();
            rapidjson::Document doc;
            Y_ENSURE(NJson::TReader::DocumentAsObject(body, doc));
            Y_ENSURE(NJson::TReader::MemberAsString(doc, "oauth_token", settings.OauthToken));
            Y_ENSURE(settings.OauthToken);
        }

        settings.RpcCodec = config.AsString(path + "/rpc_codec");
        settings.Cluster = config.AsString(path + "/cluster");
        settings.CypressDir = config.AsString(path + "/cypress_dir");
        settings.LogFile = config.AsString(path + "/log/file", "");
        settings.LogLevel = config.AsInt(path + "/log/level", 2);

        YtFactory_ = std::make_shared<NYt::TYtClientFactory>(settings);

        NYt::TTableCreatorSettings tcSettings{
            .CreateSettings = NYt::TYtClient::TCreateSettings{
                .TableTtl = TDuration::Days(config.AsInt(path + "/table_ttl__days")),
                .Codec = config.AsString(path + "/codec"),
                .ErasureCodec = "isa_lrc_12_2_2",
                .DesiredTabletCount = config.AsNum<size_t>(path + "/tablet_count"),
                .OptimizeForScan = true,
                .StoreChecksum = true,
                .MinDataTtl = 0,
                .Schema = CreateYtSchema(),
                .Timeout = TDuration::Seconds(config.AsInt(path + "/timeout__seconds")),
            },
            .PeriodType = NYt::TPeriodicTable::Day,
        };
        TableCreator_ = std::make_unique<NYt::TTableCreator>(
            YtFactory_,
            tcSettings);

        return NYt::TBatchSettings{
            .Limit = config.AsNum<size_t>(path + "/max_rows_per_transaction"),
            .Timeout = tcSettings.CreateSettings.Timeout,
        };
    }

    void TLbcbck::InitProcessor(const NXml::TConfig& config,
                                const TString& path,
                                const NYt::TBatchSettings& settings) {
        size_t workers = config.AsInt(path + "/workers");

        DataPusher_ = std::make_unique<TDataPusher>(Lb_);

        DataPusher_->CreateWorkers<TDataPushWorker, TSettings>(
            workers,
            TSettings{
                .Yt = YtFactory_,
                .BatchSettings = settings,
            });
    }

    void TLbcbck::Start() {
        Lb_->Start(
            "lbcbck_",
            [this](const NLb::TDataSet<>& data) {
                TLbParser parser;

                auto res = parser.Parse(data);

                UnistatRowsIn_ += parser.GetRowsIn();
                UnistatRowsOut_ += parser.GetRowsOut();
                UnistatErrors_ += parser.GetErrorCount();

                return res;
            });
    }

    void TLbcbck::Stop() {
        TableCreator_.reset(); // not required more
        Tvm_.reset();

        Lb_.reset(); // stop readers which push this data pipeline

        DataPusher_.reset(); // stop parsers - last users of YT
        YtFactory_.reset();  // calls NYT::Shutdown() after freeing of all pointers
    }
}
