import datetime
import os
import sys
import time

import yt.wrapper as yt
from yql.api.v1.client import YqlClient


def get_token(prefix):
    with open('./%s_token' % prefix) as f:
        return f.read()


yt = yt.YtClient(proxy='hahn', token=get_token('yt'))
yql = YqlClient(token=get_token('yql'))


def yql_query(tmp_table, old_table):
    return (
        """
use hahn;
INSERT INTO `"""
        + tmp_table
        + """` WITH TRUNCATE (
    SELECT DISTINCT
        user_ip,
        unixtime,
        def_uid,
        host_client,
        uids
    FROM ((
        SELECT
            user_ip,
            cast(unixtime as uint32) as unixtime,
            cast(uid as UINT64) as def_uid,
            (CASE
                WHEN String::StartsWith(String::SplitToList(WeakField(comment, "String", ""), ";", false, true)[0], "clid=")
                THEN String::SplitToList(String::SplitToList(WeakField(comment, "String", ""), ";", false, true)[0], "=", false, true)[1]
                ELSE ""
            END) as host_client,
            "" as uids
        FROM `"""
        + old_table
        + """`
        WHERE type == "oauthcheck" AND status == "successful"
    ) UNION ALL (
        SELECT
            WeakField(userip, "String") as user_ip,
            cast(unixtime as uint32) as unixtime,
            cast(WeakField(def_uid, "String") as UINT64) ?? 0 as def_uid,
            WeakField(host, "String") as host_client,
            String::JoinFromList(
                ListMap(
                    ListFilter(
                        ListZip(
                            String::SplitToList(WeakField(uids, "String"), ","),
                            ListReplicate(
                                WeakField(def_uid, "String", ""),
                                ListLength(String::SplitToList(WeakField(uids, "String"), ","))
                            )
                        ),
                        ($x) -> { return $x.0 != $x.1; }
                    ),
                    ($x) -> { return $x.0; }
                ),
            ",") as uids
        FROM `"""
        + old_table
        + """`
        WHERE action == "sescheck"
    ))
)"""
    )


def alter_table(name):
    return (
        """yt --proxy hahn alter-table """
        + name
        + """ --schema '<
"unique_keys" = %true;
"strict" = %true;
> [
{
    "name" = "user_ip";
    "required" = %true;
    "type" = "string";
    "sort_order" = "ascending";
};
{
    "name" = "host_client";
    "required" = %true;
    "type" = "string";
    "sort_order" = "ascending";
};
{
    "name" = "unixtime";
    "required" = %true;
    "type" = "uint32";
    "sort_order" = "ascending";
};
{
    "name" = "def_uid";
    "required" = %true;
    "type" = "uint64";
    "sort_order" = "ascending";
};
{
    "name" = "uids";
    "required" = %true;
    "type" = "string";
    "sort_order" = "ascending";
};
{
    "name" = "dummy";
    "required" = %false;
    "type" = "any";
};
]'"""
    )


TIME_TO_LIVE = datetime.timedelta(days=190)


def do(date):
    # prepare
    table_name = date.strftime('%Y-%m-%d')
    old_table = '//statbox/blackbox-log/' + table_name
    new_table = '//home/passport-rt/production/lbcbck/' + table_name
    tmp_table = new_table + '_tmp'

    expiration_time = datetime.date.today() + (date - datetime.date.today() + TIME_TO_LIVE)
    expiration_time = int(time.mktime(expiration_time.timetuple())) * 1000

    schema = [
        {"name": "user_ip", "required": True, "type": "string", "sort_order": "ascending"},
        {"name": "host_client", "required": True, "type": "string", "sort_order": "ascending"},
        {"name": "unixtime", "required": True, "type": "uint32", "sort_order": "ascending"},
        {"name": "def_uid", "required": True, "type": "uint64", "sort_order": "ascending"},
        {"name": "uids", "required": True, "type": "string", "sort_order": "ascending"},
        {"name": "dummy", "type": "any"},
    ]

    attributes = {
        'expiration_time': expiration_time,
        'compression_codec': 'brotli_8',
        'optimize_for': 'scan',
        #        'schema': schema,
        #        "dynamic": True,
    }

    # run yql
    q = yql_query(tmp_table, old_table)
    yqlr = yql.query(q, syntax_version=1)
    yqlr.run()
    print(yqlr)
    assert yqlr.is_ok

    # run yt
    yt.create("table", path=new_table, attributes=attributes, ignore_existing=True)
    os.system(alter_table(new_table))
    yt.alter_table(new_table, dynamic=False)

    yt.run_sort(
        source_table=tmp_table,
        destination_table=new_table,
        sort_by=['user_ip', 'host_client', 'unixtime', 'def_uid', 'uids'],
        spec={'force_transform': True},
    )
    yt.alter_table(new_table, dynamic=True)
    yt.mount_table(new_table, sync=True)

    # remove yql
    yt.remove(tmp_table)


if __name__ == '__main__':
    if len(sys.argv) == 2:
        x = datetime.datetime.strptime(sys.argv[1], '%Y-%m-%d').date()
        while datetime.date.today() - x < TIME_TO_LIVE:
            print('===================== starting %s' % x)
            do(x)
            x = x - datetime.timedelta(days=1)
    elif len(sys.argv) == 3:
        until = datetime.datetime.strptime(sys.argv[1], '%Y-%m-%d').date()
        x = datetime.datetime.strptime(sys.argv[2], '%Y-%m-%d').date()
        while until < x:
            print('===================== starting %s' % x)
            do(x)
            x = x - datetime.timedelta(days=1)
    else:
        raise Exception('Usage: script date [date]')
