#include <passport/infra/daemons/lbcbck/src/lb_parser.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/draft/date.h>
#include <util/system/env.h>

using namespace NPassport;
using namespace NPassport::NLbcbck;

Y_UNIT_TEST_SUITE(LbParser) {
    static void SetTimeZone() {
        SetEnv("TZ", "Europe/Moscow");
        tzset();
    }

    class TLbParserTest: public TLbParser {
    public:
        using TLbParser::GetConverter;
        using TLbParser::ParseAsOauthcheck;
        using TLbParser::ParseAsSescheck;
        using TLbParser::ParseRows;
        using TLbParser::Result_;
        using TLbParser::TLbParser;
        using TLbParser::TryParse;
    };

    Y_UNIT_TEST(rowCtx) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(TLbParser::TRowCtx("asdas"),
                                       yexception,
                                       "string is not tskv");

        TLbParser::TRowCtx ctx("tskv\t"
                               "tskv_format=some_format\t"
                               "action=some_action\t"
                               "status=some_status\t"
                               "userip=some_userip\t"
                               "user_ip=some_userip\t"
                               "unixtime=some_unixtime\t"
                               "host=some_host\t"
                               "def_uid=some_def_uid\t"
                               "uids=some_uids\t"
                               "uid=some_uid\t"
                               "user_port=some_user_port\t"
                               "client_id=some_client_id\t"
                               "lol=kek\t");

        UNIT_ASSERT_VALUES_EQUAL("some_format", ctx.Get(TLbParser::TRowCtx::EKey::Format));
        UNIT_ASSERT_VALUES_EQUAL("some_action", ctx.Get(TLbParser::TRowCtx::EKey::Action));
        UNIT_ASSERT_VALUES_EQUAL("some_status", ctx.Get(TLbParser::TRowCtx::EKey::Status));
        UNIT_ASSERT_VALUES_EQUAL("some_userip", ctx.Get(TLbParser::TRowCtx::EKey::UserIp));
        UNIT_ASSERT_VALUES_EQUAL("some_unixtime", ctx.Get(TLbParser::TRowCtx::EKey::Unixtime));
        UNIT_ASSERT_VALUES_EQUAL("some_host", ctx.Get(TLbParser::TRowCtx::EKey::Host));
        UNIT_ASSERT_VALUES_EQUAL("some_def_uid", ctx.Get(TLbParser::TRowCtx::EKey::DefUid));
        UNIT_ASSERT_VALUES_EQUAL("some_uids", ctx.Get(TLbParser::TRowCtx::EKey::Uids));
        UNIT_ASSERT_VALUES_EQUAL("some_uid", ctx.Get(TLbParser::TRowCtx::EKey::Uid));
        UNIT_ASSERT_VALUES_EQUAL("some_user_port", ctx.Get(TLbParser::TRowCtx::EKey::UserPort));
        UNIT_ASSERT_VALUES_EQUAL("some_client_id", ctx.Get(TLbParser::TRowCtx::EKey::ClientId));
    }

    Y_UNIT_TEST(sescheck) {
        SetTimeZone();

        TWritableRow row;

        TLbParser::TRowCtx ctx("tskv\t"
                               "userip=val4\t"
                               "unixtime=val6\t"
                               "host=val7\t"
                               "def_uid=127\t"
                               "uids=val9\t");
        UNIT_ASSERT_EXCEPTION_CONTAINS(TLbParserTest::ParseAsSescheck(ctx),
                                       yexception,
                                       "'unixtime' must be int");

        ctx = TLbParser::TRowCtx("tskv\t"
                                 "userip=val4\t"
                                 "unixtime=100500\t"
                                 "host=val7\t"
                                 "def_uid=val8\t"
                                 "uids=val9\t");
        UNIT_ASSERT_EXCEPTION_CONTAINS(TLbParserTest::ParseAsSescheck(ctx),
                                       yexception,
                                       "'def_uid' must be int");

        ctx = TLbParser::TRowCtx("tskv\t"
                                 "userip=val4\t"
                                 "unixtime=100500\t"
                                 "host=val7\t"
                                 "def_uid=\t"
                                 "uids=val9\t");

        row = TLbParserTest::ParseAsSescheck(ctx);
        UNIT_ASSERT_VALUES_EQUAL("val4", row.UserIp);
        UNIT_ASSERT_VALUES_EQUAL("val7", row.HostClient);
        UNIT_ASSERT_VALUES_EQUAL(100500, row.Unixtime);
        UNIT_ASSERT_VALUES_EQUAL(0, row.DefUid);
        UNIT_ASSERT_VALUES_EQUAL("val9", row.Uids);
        UNIT_ASSERT_VALUES_EQUAL(0, row.UserPort);

        ctx = TLbParser::TRowCtx("tskv\t"
                                 "userip=val4\t"
                                 "unixtime=100500\t"
                                 "host=val7\t"
                                 "def_uid=127\t"
                                 "user_port=178\t"
                                 "uids=127,49,49,,127\t");

        row = TLbParserTest::ParseAsSescheck(ctx);
        UNIT_ASSERT_VALUES_EQUAL("val4", row.UserIp);
        UNIT_ASSERT_VALUES_EQUAL("val7", row.HostClient);
        UNIT_ASSERT_VALUES_EQUAL(100500, row.Unixtime);
        UNIT_ASSERT_VALUES_EQUAL(127, row.DefUid);
        UNIT_ASSERT_VALUES_EQUAL("49,49", row.Uids);
        UNIT_ASSERT_VALUES_EQUAL(178, row.UserPort);
    }

    Y_UNIT_TEST(oauthcheck) {
        SetTimeZone();

        std::optional<TWritableRow> row;

        TLbParser::TRowCtx ctx("tskv\t"
                               "user_ip=val5\t"
                               "unixtime=val6\t"
                               "uid=127\t");
        UNIT_ASSERT_EXCEPTION_CONTAINS(TLbParserTest::ParseAsOauthcheck(ctx),
                                       yexception,
                                       "'unixtime' must be int");

        ctx = TLbParser::TRowCtx("tskv\t"
                                 "user_ip=val5\t"
                                 "unixtime=100500\t"
                                 "uid=val8\t");
        UNIT_ASSERT_EXCEPTION_CONTAINS(TLbParserTest::ParseAsOauthcheck(ctx),
                                       yexception,
                                       "'uid' must be int");

        ctx = TLbParser::TRowCtx("tskv\t"
                                 "user_ip=val5\t"
                                 "unixtime=100500\t");
        UNIT_ASSERT(!TLbParserTest::ParseAsOauthcheck(ctx));

        ctx = TLbParser::TRowCtx("tskv\t"
                                 "user_ip=val5\t"
                                 "unixtime=100500\t"
                                 "user_port=kekw\t"
                                 "uid=127\t");

        row = TLbParserTest::ParseAsOauthcheck(ctx);
        UNIT_ASSERT(row);
        UNIT_ASSERT_VALUES_EQUAL("val5", row->UserIp);
        UNIT_ASSERT_VALUES_EQUAL("", row->HostClient);
        UNIT_ASSERT_VALUES_EQUAL(100500, row->Unixtime);
        UNIT_ASSERT_VALUES_EQUAL(127, row->DefUid);
        UNIT_ASSERT_VALUES_EQUAL("", row->Uids);
        UNIT_ASSERT_VALUES_EQUAL(0, row->UserPort);

        ctx = TLbParser::TRowCtx("tskv\t"
                                 "user_ip=val5\t"
                                 "unixtime=100500\t"
                                 "user_port=178\t"
                                 "client_id=idbaiasndiad\t"
                                 "uid=127\t");

        row = TLbParserTest::ParseAsOauthcheck(ctx);
        UNIT_ASSERT(row);
        UNIT_ASSERT_VALUES_EQUAL("val5", row->UserIp);
        UNIT_ASSERT_VALUES_EQUAL("idbaiasndiad", row->HostClient);
        UNIT_ASSERT_VALUES_EQUAL(100500, row->Unixtime);
        UNIT_ASSERT_VALUES_EQUAL(127, row->DefUid);
        UNIT_ASSERT_VALUES_EQUAL("", row->Uids);
        UNIT_ASSERT_VALUES_EQUAL(178, row->UserPort);
    }

    Y_UNIT_TEST(tryParseAsOauth) {
        std::optional<TWritableRow> row;
        row = TLbParserTest::TryParse("tskv\t"
                                      "user_ip=val5\t"
                                      "unixtime=100500\t"
                                      "client_id=idbaiasndiad\t"
                                      "uid=127\t");
        UNIT_ASSERT(!row);

        row = TLbParserTest::TryParse("tskv\t"
                                      "tskv_format=oauth-log\t"
                                      "user_ip=val5\t"
                                      "unixtime=100500\t"
                                      "client_id=idbaiasndiad\t"
                                      "uid=127\t"
                                      "status=ERROR\t");
        UNIT_ASSERT(!row);

        row = TLbParserTest::TryParse("tskv\t"
                                      "tskv_format=oauth-log\t"
                                      "user_ip=val5\t"
                                      "unixtime=100500\t"
                                      "client_id=idbaiasndiad\t"
                                      "uid=127\t"
                                      "status=OK\t");
        UNIT_ASSERT(row);
        UNIT_ASSERT_VALUES_EQUAL("idbaiasndiad", row->HostClient);
    }

    Y_UNIT_TEST(tryParseAsSess) {
        std::optional<TWritableRow> row;

        row = TLbParserTest::TryParse("tskv\t"
                                      "userip=val4\t"
                                      "host=val7\t"
                                      "unixtime=100500\t"
                                      "uids=127\t");
        UNIT_ASSERT(!row);

        row = TLbParserTest::TryParse("tskv\t"
                                      "tskv_format=blackbox-log\t"
                                      "userip=val4\t"
                                      "host=val7\t"
                                      "unixtime=100500\t"
                                      "uids=127\t"
                                      "action=sess");
        UNIT_ASSERT(!row);

        row = TLbParserTest::TryParse("tskv\t"
                                      "tskv_format=blackbox-log\t"
                                      "userip=val4\t"
                                      "host=val7\t"
                                      "unixtime=100500\t"
                                      "uids=127\t"
                                      "action=sescheck");
        UNIT_ASSERT(row);
        UNIT_ASSERT_VALUES_EQUAL("val7", row->HostClient);
    }

    Y_UNIT_TEST(getStorage) {
        TLbParserTest parser;
        TLbParser::TResult res;

        UNIT_ASSERT_VALUES_EQUAL(0, res.ByTable.size());

        TWritableRow row1;
        row1.Unixtime = 100500;

        TLbParserTest::GetConverter(res, row1, 10);
        UNIT_ASSERT_VALUES_EQUAL(1, res.ByTable.size());

        TWritableRow row2;
        row2.Unixtime = 10005000;
        TLbParserTest::GetConverter(res, row2, 10);
        UNIT_ASSERT_VALUES_EQUAL(2, res.ByTable.size());
        UNIT_ASSERT(res.ByTable.find(100500) != res.ByTable.end());
        UNIT_ASSERT(res.ByTable.find(10005000) != res.ByTable.end());
    }

    Y_UNIT_TEST(parseRows) {
        NLb::TData data;
        NLb::TTopicData td;
        td.Data.push_back(NLb::TChunk{
            .Data = "tskv\t"
                    "tskv_format=oauth-log\t"
                    "user_ip=val5\t"
                    "unixtime=100500\t"
                    "client_id=deleted client\t"
                    "uid=127\t"
                    "status=ERROR\n"

                    "tskv\t"
                    "tskv_format=oauth-log\t"
                    "user_ip=val5\t"
                    "unixtime=100500\t"
                    "client_id=idbaiasndiad\t"
                    "uid=127\t"
                    "status=OK\n"

                    "tskv\t"
                    "tskv_format=oauth-log\t"
                    "user_ip=val5\t"
                    "unixtime=100500\t"
                    "client_id=idbaiasndiad\t"
                    "uid=127\t"
                    "status=OK\n"

                    "tskv\t"
                    "tskv_format=oauth-log\t"
                    "user_ip=val5\t"
                    "unixtime=1005000\t"
                    "client_id=idbaiasndiad\t"
                    "uid=127\t"
                    "status=OK\n"

                    "tskv\t"
                    "tskv_format=blackbox-log\t"
                    "userip=val4\t"
                    "host=val7\t"
                    "unixtime=100500\t"
                    "uids=127\t"
                    "action=sescheck\n",
        });
        data.Messages.push_back(std::move(td));

        TLbParserTest parser;
        parser.ParseRows(data);
        TLbParser::TResult res = std::move(parser.Result_);

        UNIT_ASSERT_VALUES_EQUAL(2, res.ByTable.size());

        auto it = res.ByTable.find(100500);
        UNIT_ASSERT(it != res.ByTable.end());
        UNIT_ASSERT_VALUES_EQUAL("1970-01-02", it->first.TableName());
        UNIT_ASSERT_VALUES_EQUAL(2, it->second->size());
        UNIT_ASSERT_VALUES_EQUAL("userip=val5. host_client=idbaiasndiad. unixtime=100500. def_uid=127. uids=. port=0\n#"
                                 "userip=val4. host_client=val7. unixtime=100500. def_uid=0. uids=127. port=0\n#",
                                 it->second->DebugString());

        it = res.ByTable.find(1005000);
        UNIT_ASSERT(it != res.ByTable.end());
        UNIT_ASSERT_VALUES_EQUAL("1970-01-12", it->first.TableName());
        UNIT_ASSERT_VALUES_EQUAL(1, it->second->size());
        UNIT_ASSERT_VALUES_EQUAL("userip=val5. host_client=idbaiasndiad. unixtime=1005000. def_uid=127. uids=. port=0\n#",
                                 it->second->DebugString());
    }
}
