import json
import jsondiff
import os
import requests
import time
import yatest.common as yc
import yt.wrapper

from passport.infra.recipes.common import log, start_daemon
from yatest.common import network

import passport.infra.daemons.lbcbck.ut_medium.logbroker as lb

from passport.infra.daemons.lbcbck.ut_medium.helpers import (
    deep_sorted,
    pretty_json,
    test_with_deadline,
)


class LbcbckFixture:
    logbroker_client_id = "test_client"

    def __init__(self, lb_port=None, yt_proxy=None, timeout=5):
        self._pm = network.PortManager()

        self.http_port = self._pm.get_tcp_port()

        output_path = yc.output_path('lbcbck')
        if not os.path.isdir(output_path):
            os.mkdir(output_path)

        logs_dir = os.path.join(output_path, 'logs')
        if not os.path.isdir(logs_dir):
            os.mkdir(logs_dir)

        oauth_token_path = os.path.join(output_path, 'yt.json')
        config_path = os.path.join(output_path, 'config.json')

        lbcbck_config = f"""<?xml version="1.0"?>
<config>
    <http_daemon>
        <listen_address>localhost</listen_address>
        <port>{self.http_port}</port>
    </http_daemon>
    <component>
        <logger>
            <file>{os.path.join(logs_dir, "lbcbck.log")}</file>
            <level>DEBUG</level>
            <print-level>yes</print-level>
            <time-format>_DEFAULT_</time-format>
        </logger>

        <processor>
            <workers>128</workers>
        </processor>

        <tvmtool>
            <port>{open(yc.work_path("tvmtool.port")).read()}</port>
            <authtoken>{yc.work_path("tvmtool.authtoken")}</authtoken>
        </tvmtool>

        <yt>
            <cypress_dir>//home</cypress_dir>
            <table_ttl__days>2</table_ttl__days>

            <oauth_token_file>{oauth_token_path}</oauth_token_file>
            <cluster>{yt_proxy if yt_proxy else os.environ["YT_PROXY"]}</cluster>
            <log_level>3</log_level>
            <timeout__seconds>10</timeout__seconds>
            <max_rows_per_transaction>5000</max_rows_per_transaction>
            <codec>lz4</codec>
            <tablet_count>100</tablet_count>
            <log>
                <file>{os.path.join(logs_dir, "yt.log")}</file>
                <level>3</level> <!--1(Trace)...7(Fatal)-->
            </log>

            <!--https://a.yandex-team.ru/arc/trunk/arcadia/yt/yt/core/compression/public.h?rev=7734908#L12-->
            <!--here must be values with snake case-->
            <rpc_codec>lz4</rpc_codec>
        </yt>

        <logbroker>
            <client_id>{self.logbroker_client_id}</client_id>
            <port>{lb_port if lb_port else lb.port()}</port>

            <queue_size>512</queue_size>
            <queue_workers>30</queue_workers>
            <memory_usage_per_server__mb>100</memory_usage_per_server__mb>
            <read_max_bytes__mb>10</read_max_bytes__mb>
            <read_max_count>4096</read_max_count>
            <max_uncommitted_size>167772160</max_uncommitted_size>

            <log_level>6</log_level>
            <topics>
                <blackbox-log>blackbox-log</blackbox-log>
                <oauth-log>oauth-log</oauth-log>
            </topics>
            <servers>
                <server>localhost</server>
            </servers>
        </logbroker>
    </component>
</config>
"""

        with open(oauth_token_path, 'w') as f:
            f.write('{"oauth_token":"token"}')

        with open(config_path, 'w') as f:
            f.write(lbcbck_config)

        command = [
            yc.build_path('passport/infra/daemons/lbcbck/daemon/lbcbck'),
            '-c',
            config_path,
        ]
        self._process = start_daemon(command, os.environ.copy(), self.http_port, 300)
        self._deadline = time.time() + timeout

        yt_config = yt.wrapper.default_config.get_config_from_env()
        yt_config["proxy"]["retries"]["backoff"] = {"policy": "constant_time", "constant_time": 500}
        self._ytc = yt.wrapper.YtClient(config=yt_config)

    def stop(self):
        log("Closing lbcbck fixture with %d sec. before deadline" % (self._deadline - time.time()))

        self._pm.release()
        self._process.terminate()

    def ping(self, code=200, resp='OK\n'):
        res = requests.get('http://127.0.0.1:%d/ping' % self.http_port)

        assert res.status_code == code
        assert res.text == resp

    def check_yt_content(self, content, preproc=lambda x: x):
        for table, expected_rows in content.items():
            expected_rows = deep_sorted(expected_rows)

            def check_table_content():
                assert self._ytc.exists(table), table
                assert self._ytc.get_attribute(table, "dynamic")
                actual_rows = [preproc(r) for r in self._ytc.select_rows("* FROM[{table}]".format(table=table))]
                actual_rows = deep_sorted(actual_rows)

                assert expected_rows == actual_rows, "%s:\nGOT: %s\nEXP: %s\nDIFF: %s\n" % (
                    table,
                    json.dumps(actual_rows, indent=2),
                    json.dumps(expected_rows, indent=2),
                    pretty_json(jsondiff.diff(expected_rows, actual_rows, dump=True)),
                )

            test_with_deadline(check_table_content, self._deadline)
