import datetime
import time
import yt.wrapper

TABLES_DIR = "//home"


class DailyTable:
    def __init__(self, ts=None):
        if ts is None:
            ts = int(time.time())
        self._date = self.__get_day_start_by_ts(ts)

    @staticmethod
    def __get_day_start_by_ts(ts):
        date = datetime.datetime.fromtimestamp(ts)
        return datetime.datetime(year=date.year, month=date.month, day=date.day)

    def date(self):
        return self._date

    def ts(self):
        return int(self._date.timestamp())

    def name(self):
        return self._date.strftime("%Y-%m-%d")

    def path(self, directory):
        return '/'.join((directory, self.name()))

    def prev(self):
        return DailyTable(self.ts() - 1)

    def next(self):
        return DailyTable(self.ts() + 60 * 60 * 24)


def create_dir(dir):
    if yt.wrapper.exists(dir):
        yt.wrapper.remove(dir, recursive=True)

    yt.wrapper.create("map_node", dir, recursive=True)


def create_table(table_name, schema, atomicity="none", **attributes):
    if yt.wrapper.exists(table_name):
        yt.wrapper.remove(table_name)

    yt.wrapper.create(
        "table",
        table_name,
        attributes={
            "dynamic": True,
            "schema": schema,
            "atomicity": atomicity,
            **attributes,
        },
        recursive=True,
    )
    yt.wrapper.mount_table(table_name)


def create_daily_tables(directory, schema, n=3, ts=None, **attributes):
    create_dir(directory)

    daily_table = DailyTable(ts).next()
    for _ in range(n + 1):
        create_table(daily_table.path(directory), schema, **attributes)
        daily_table = daily_table.prev()


def prepare_yt():
    create_dir(TABLES_DIR)


def create_tables(n=3):
    schema = [
        {
            "name": "user_ip",
            "required": True,
            "sort_order": "ascending",
            "type_v3": "string",
        },
        {
            "name": "host_client",
            "required": True,
            "sort_order": "ascending",
            "type_v3": "string",
        },
        {
            "name": "unixtime",
            "required": True,
            "sort_order": "ascending",
            "type_v3": "uint32",
        },
        {
            "name": "def_uid",
            "required": True,
            "sort_order": "ascending",
            "type_v3": "uint64",
        },
        {
            "name": "uids",
            "required": True,
            "sort_order": "ascending",
            "type_v3": "string",
        },
        {
            "name": "user_port",
            "required": False,
            "sort_order": "ascending",
            "type_v3": {"type_name": "optional", "item": "uint16"},
        },
        {
            "name": "dummy",
            "required": False,
            "type_v3": {"type_name": "optional", "item": "yson"},
        },
    ]

    create_daily_tables(TABLES_DIR, schema, n)
