import json
import sys

# https://lb.yandex-team.ru/docs/concepts/clusters_and_installations
# https://lb.yandex-team.ru/docs/concepts/security

SAS = "sas"
IVA = "iva"
MAN = "man"


def gen_hosts(prefix, number):
    return json.dumps(['%s%d.passport.yandex.net' % (prefix, i + 1) for i in range(number)])


def gen_dbpool_hosts(prefix, number):
    return json.dumps([{"host": '%s%d.passport.yandex.net' % (prefix, i + 1)} for i in range(number)])


SETTINGS = {
    "development": {
        "READ_MAX_BYTES": 8 * 1024,
        "MAX_UNCOMMITTED_SIZE": 1024 * 1024,
        "SELF_TVM_ID": "2019661",
        "LOGBROKER_CLIENT_ID": "historydb2/dev/lbchdb",
        "LOGBROKER_SERVER": "sas.logbroker.yandex.net",
        "LOGBROKER_TOPIC__AUTH": "historydb2/testing/auth",
        "LOGBROKER_TOPIC__BB_AUTH": "historydb2/testing/bb_auth",
        "LOGBROKER_TOPIC__EVENT": "historydb2/testing/event",
        "LOGBROKER_TOPIC__MAIL_USER_JOURNAL": "historydb2/testing/mail_user_journal",
        "LOGBROKER_TOPIC__OAUTH": "historydb2/testing/oauth",
        "LOGBROKER_TOPIC__RESTORE": "historydb2/testing/restore",
        "LOGBROKER_TOPIC__YASMS_PRIVATE": "historydb2/testing/yasms_private",
        "LOGBROKER_TOPIC__PUSH": "historydb2/testing/push",
        "LOGBROKER_TOPIC__PUSH_SUBSCRIPTION": "historydb2/testing/push_subscription",
        "HBASE_HOSTS": gen_hosts("kvs-test-m", 3),
        "HBASE_TABLES_PREFIX": "__dev_",
        "PUSH_WORKERS": 8,
        "BLACKBOX_TVM_ID": 239,
        "BLACKBOX_HOSTS": '[{"host": "https://blackbox-mimino.yandex.net"}]',
        "KOLMOGOR_TVM_ID": "2000130",
        "KOLMOGOR_PORT": 11080,
        "KOLMOGOR_HOSTS": '[{"host": "::1"}]',
        "YT_CLUSTER": "seneca-sas",
        "YT_DIR": "//home/passport-hdb/development/historydb",
        "ENABLE_LASTAUTH_TO_HBASE": "false",
        "ENABLE_MAIL_TO_YT": "true",
        "ENABLE_MAIL_TO_HBASE": "false",
        "SENDER_SAMPLE_RATIO": 5,
        "TABLE_TTL_USER_HISTORY": 5356800,
        "TABLE_TTL_CORP_USER_HISTORY": 2678400,
        "TABLE_TTL_PUSH": 2678400,
        "TABLE_TTL_AUTHS": 2678400,
    },
    "testing": {
        "READ_MAX_BYTES": 512 * 1024,
        "MAX_UNCOMMITTED_SIZE": 160 * 1024 * 1024,
        "SELF_TVM_ID": "2020010",
        "LOGBROKER_CLIENT_ID": "historydb2/testing/lbchdb",
        "LOGBROKER_SERVER": "sas.logbroker.yandex.net",
        "LOGBROKER_TOPIC__AUTH": "historydb2/testing/auth",
        "LOGBROKER_TOPIC__BB_AUTH": "historydb2/testing/bb_auth",
        "LOGBROKER_TOPIC__EVENT": "historydb2/testing/event",
        "LOGBROKER_TOPIC__MAIL_USER_JOURNAL": "historydb2/testing/mail_user_journal",
        "LOGBROKER_TOPIC__OAUTH": "historydb2/testing/oauth",
        "LOGBROKER_TOPIC__RESTORE": "historydb2/testing/restore",
        "LOGBROKER_TOPIC__YASMS_PRIVATE": "historydb2/testing/yasms_private",
        "LOGBROKER_TOPIC__PUSH": "historydb2/testing/push",
        "LOGBROKER_TOPIC__PUSH_SUBSCRIPTION": "historydb2/testing/push_subscription",
        "HBASE_HOSTS": gen_hosts("kvs-test-m", 5),
        "HBASE_TABLES_PREFIX": "",
        "PUSH_WORKERS": 8,
        "BLACKBOX_TVM_ID": 239,
        "BLACKBOX_HOSTS": '[{"host": "https://blackbox-mimino.yandex.net"}]',
        "KOLMOGOR_TVM_ID": "2025622",
        "KOLMOGOR_PORT": 8710,
        "KOLMOGOR_HOSTS": '[{"host": "::1"}]',
        "YT_CLUSTER": "seneca-vla",
        "YT_DIR": "//home/passport-hdb/testing/historydb",
        "ENABLE_LASTAUTH_TO_HBASE": "false",
        "ENABLE_MAIL_TO_YT": "true",
        "ENABLE_MAIL_TO_HBASE": "false",
        "SENDER_SAMPLE_RATIO": 5,
        "TABLE_TTL_USER_HISTORY": 5356800,
        "TABLE_TTL_CORP_USER_HISTORY": 2678400,
        "TABLE_TTL_PUSH": 2678400,
        "TABLE_TTL_AUTHS": 5356800,
    },
    "production": {
        "READ_MAX_BYTES": 512 * 1024,
        "MAX_UNCOMMITTED_SIZE": 1000 * 1024 * 1024,
        "SELF_TVM_ID": "2020012",
        "LOGBROKER_CLIENT_ID": "historydb2/production/lbchdb",
        "LOGBROKER_SERVER": {
            SAS: "sas.logbroker.yandex.net",
            IVA: "vla.logbroker.yandex.net",
            MAN: "man.logbroker.yandex.net",
        },
        "LOGBROKER_TOPIC__AUTH": "historydb2/production/auth",
        "LOGBROKER_TOPIC__BB_AUTH": "historydb2/production/bb_auth",
        "LOGBROKER_TOPIC__EVENT": "historydb2/production/event",
        "LOGBROKER_TOPIC__MAIL_USER_JOURNAL": "historydb2/production/mail_user_journal",
        "LOGBROKER_TOPIC__OAUTH": "historydb2/production/oauth",
        "LOGBROKER_TOPIC__RESTORE": "historydb2/production/restore",
        "LOGBROKER_TOPIC__YASMS_PRIVATE": "historydb2/production/yasms_private",
        "LOGBROKER_TOPIC__PUSH": "historydb2/production/push",
        "LOGBROKER_TOPIC__PUSH_SUBSCRIPTION": "historydb2/production/push_subscription",
        "HBASE_HOSTS": {
            SAS: gen_hosts("kvs-s", 15),
            IVA: gen_hosts("kvs-i", 15),
            MAN: gen_hosts("kvs-f", 20),
        },
        "HBASE_TABLES_PREFIX": "",
        "PUSH_WORKERS": {
            SAS: 400,
            IVA: 400,
            MAN: 400,
        },
        "BLACKBOX_TVM_ID": 222,
        "BLACKBOX_HOSTS": '[{"host": "https://blackbox.yandex.net"}]',
        "KOLMOGOR_TVM_ID": "2025620",
        "KOLMOGOR_PORT": 8710,
        "KOLMOGOR_HOSTS": {
            SAS: gen_dbpool_hosts("kvs-api-s", 4),
            IVA: gen_dbpool_hosts("kvs-api-i", 4),
            MAN: gen_dbpool_hosts("kvs-api-f", 4),
        },
        "YT_CLUSTER": {
            SAS: "seneca-sas",
            IVA: "seneca-vla",
            MAN: "seneca-man",
        },
        "YT_DIR": "//home/passport-hdb/production/historydb",
        "ENABLE_LASTAUTH_TO_HBASE": {
            SAS: "false",
            IVA: "false",
            MAN: "true",
        },
        "ENABLE_MAIL_TO_YT": "true",
        "ENABLE_MAIL_TO_HBASE": {
            SAS: "false",
            IVA: "false",
            MAN: "true",
        },
        "SENDER_SAMPLE_RATIO": 100,
        "TABLE_TTL_USER_HISTORY": 41708800,
        "TABLE_TTL_CORP_USER_HISTORY": 18408200,
        "TABLE_TTL_PUSH": 16070400,
        "TABLE_TTL_AUTHS": 64281600,
    },
}


def do(template, env, dc):
    if env not in SETTINGS:
        raise Exception("env '%s' is unknown" % env)

    setts = SETTINGS[env]
    for key, value in setts.items():
        if isinstance(value, dict):
            value = value[dc]
        template = template.replace('<<%s>>' % key, str(value))

    if '<<' in template or '>>' in template:
        raise Exception("some options were not specified: %s\n" % template)

    print(template)


def get_dc(hostname):
    if hostname.startswith('kvs-loader-s'):
        return SAS
    if hostname.startswith('kvs-loader-f'):
        return MAN
    if hostname.startswith('kvs-loader-i'):
        return IVA

    if 'dev' in hostname or 'test' in hostname:
        return None

    raise Exception('Unconfigured DC for host: %s' % hostname)


if __name__ == '__main__':
    if len(sys.argv) != 4:
        print("Usage: %s <template_file> <env> <hostname>" % (sys.argv[0]))
        sys.exit(1)

    with open(sys.argv[1]) as f:
        template = f.read()
        assert len(template) > 0

    do(template, sys.argv[2], get_dc(sys.argv[3]))
