#include "blackbox_client.h"

#include <passport/infra/libs/cpp/idn/idn.h>
#include <passport/infra/libs/cpp/json/reader.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <contrib/libs/rapidjson/include/rapidjson/pointer.h>

namespace NPassport::NLbchdb {
    TBlackboxClient::TBlackboxClient(NDbPool::TDbPool& bb)
        : Bb_(bb)
    {
    }

    TBlackboxClient::TRequestHandle TBlackboxClient::SendRequest(const TRequest& req) {
        NDbPool::TQuery q(req.Path);
        q.SetHttpBody(TString(req.Body));

        NDbPool::TNonBlockingHandle h(Bb_);
        h.SendQuery(std::move(q));

        return {
            .Req = req,
            .Handle = std::move(h),
        };
    }

    TString TBlackboxClient::WaitResponse(TBlackboxClient::TRequestHandle& req) const {
        NDbPool::TNonBlockingHandle h = std::move(req.Handle);

        NDbPool::THttpResponse resp = h.WaitResult()->ToHttpResponse();
        Y_ENSURE(resp.Status == 200,
                 "bad code from blackbox: " << resp.Status << ". " << resp.Body);

        return std::move(resp.Body);
    }

    TBlackboxClient::TRequest TBlackboxClient::CreateUserinfoRequest(const TStringBuf email,
                                                                     const TString& path) {
        return TRequest{
            .Path = path,
            .Body = NUtils::CreateStr(
                "method=userinfo"
                "&format=json"
                "&userip=127.0.0.1"
                "&sid=smtp"
                "&login=",
                NUtils::Urlencode(email)),
        };
    }

    TBlackboxClient::TRequest TBlackboxClient::CreateEmailBindingsRequest(TStringBuf email,
                                                                          const TString& path) {
        // normalize for passport db
        TStringBuf login = email.NextTok('@');
        TStringBuf domain = email;

        const TString normalized = NUtils::CreateStr(
            login,
            "@",
            NIdn::UtfToPunycode(TString(domain)));

        return TRequest{
            .Path = path,
            .Body = NUtils::CreateStr(
                "method=email_bindings"
                "&format=json"
                "&email=",
                NUtils::Urlencode(normalized)),
        };
    }

    static void CheckException(rapidjson::Document& doc) {
        const rapidjson::Value* exc;
        if (!NJson::TReader::MemberAsObject(doc, "exception", exc)) {
            return;
        }

        TStringBuf status;
        TStringBuf err;
        NJson::TReader::MemberAsString(*exc, "value", status);
        NJson::TReader::MemberAsString(doc, "error", err);

        ythrow TBlackboxClient::TParseException()
            << "blackbox exception (" << status << "): " << err;
    }

    static ui64 CastUidFromString(const rapidjson::Value* uid) {
        Y_ENSURE_EX(uid->IsString(),
                    TBlackboxClient::TParseException() << "uid is not string");

        TStringBuf uidBuf(uid->GetString(), uid->GetStringLength());

        ui64 res = 0;
        Y_ENSURE_EX(TryIntFromString<10>(uidBuf, res),
                    TBlackboxClient::TParseException() << "failed to parse uid: '" << uidBuf << "'");

        return res;
    }

    static const rapidjson::Pointer JSON_POINTER_USERINFO("/users/0/uid/value");

    std::optional<ui64> TBlackboxClient::ParseResponseForUserinfo(const TStringBuf response) {
        rapidjson::Document doc;
        Y_ENSURE_EX(NJson::TReader::DocumentAsObject(response, doc),
                    TParseException() << "malformed json from blackbox");

        CheckException(doc);

        const rapidjson::Value* uid = JSON_POINTER_USERINFO.Get(doc);
        if (!uid) {
            return {};
        }

        return CastUidFromString(uid);
    }

    std::vector<ui64> TBlackboxClient::ParseResponseForEmailBindings(const TStringBuf response) {
        rapidjson::Document doc;
        Y_ENSURE_EX(NJson::TReader::DocumentAsObject(response, doc),
                    TParseException() << "malformed json from blackbox");

        CheckException(doc);

        const rapidjson::Value* arr = nullptr;
        Y_ENSURE_EX(NJson::TReader::MemberAsArray(doc, "uids", arr),
                    TParseException() << "missing uid array");

        std::vector<ui64> res;
        res.reserve(arr->Size());

        for (auto it = arr->Begin(); it != arr->End(); ++it) {
            res.push_back(CastUidFromString(it));
        }

        return res;
    }
}
