#include "decryptor.h"

#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NLbchdb::NCrypto {
    TDecryptor::TDecryptor(TKeyRingPtr keyring)
        : Keyring_(std::move(keyring))
    {
    }

    static void Parse(TStringBuf encrypted,
                      NUtils::TCrypto::TCiphertext& crypt,
                      TStringBuf& associatedDataBuf,
                      ui64& keyId) {
        const char delim = ':';

        const TStringBuf ivBuf = encrypted.NextTok(delim);
        const TStringBuf ciphertextBuf = encrypted.NextTok(delim);
        associatedDataBuf = encrypted.NextTok(delim);
        const TStringBuf authTagBuf = encrypted.NextTok(delim);

        Y_ENSURE(ivBuf && ciphertextBuf && associatedDataBuf && authTagBuf && !encrypted,
                 "Decryptor: invalid format of encrypted string");

        crypt.Iv = NUtils::Base64ToBin(ivBuf);
        crypt.Text = NUtils::Base64ToBin(ciphertextBuf);
        crypt.Tag = NUtils::Base64ToBin(authTagBuf);

        Y_ENSURE(crypt.Iv && crypt.Text && crypt.Tag,
                 "Decryptor: invalid base64 in encrypted string");

        TStringBuf keyIdBuf = TStringBuf(associatedDataBuf);
        keyIdBuf.NextTok('=');
        Y_ENSURE(TryIntFromString<10>(keyIdBuf, keyId),
                 "Decryptor: invalid key id in encrypted string: '" << keyIdBuf << "'");
    }

    TString TDecryptor::Decrypt(TStringBuf encrypted) const {
        NUtils::TCrypto::TCiphertext crypt;
        ui64 keyId = 0;
        TStringBuf associatedDataBuf;
        Parse(encrypted, crypt, associatedDataBuf, keyId);

        TString res;
        TString err;
        Y_ENSURE(NUtils::TCrypto::DecryptGcm(
                     GetKey(keyId),
                     crypt,
                     res,
                     associatedDataBuf,
                     &err),
                 "Decryptor: failed to decrypt with aes gcm: " << err);

        return res;
    }

    const TString& TDecryptor::GetKey(ui64 keyId) const {
        try {
            return Keyring_->GetKey(keyId);
        } catch (const std::exception& e) {
            throw yexception() << "Decryptor: failed to get key: " << e.what();
        }
    }
}
