#include "key_ring.h"

#include <passport/infra/libs/cpp/json/config.h>
#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/system/fs.h>

namespace NPassport::NLbchdb::NCrypto {
    const TString IKeyRing::EMPTY_KEY = "";

    TKeyRingHolderWithDefaultKey::TKeyRingHolderWithDefaultKey(TKeyRingPtr keyRing, ui64 defaultKeyID)
        : KeyRing_(std::move(keyRing))
    {
        Y_ENSURE(KeyRing_, "No keys");
        SetDefaultKeyID(defaultKeyID);
    }

    void TKeyRingHolderWithDefaultKey::SetDefaultKeyID(ui64 keyID) {
        Y_ENSURE(KeyRing_->HasKey(keyID), "No key for default key id '" << keyID << "'");
        DefaultKeyID_ = keyID;
    }

    TSimpleKeyRing::TSimpleKeyRing(TSimpleKeyRing::TKeys&& keys)
        : Keys_(std::move(keys))
    {
    }

    TSimpleKeyRing::TSimpleKeyRing(const TString& filePath)
        : TSimpleKeyRing(ReadFromFile(filePath))
    {
    }

    TSimpleKeyRing::TKeys TSimpleKeyRing::ReadFromFile(const TString& filePath) {
        NJson::TConfig config = NJson::TConfig::ReadFromFile(filePath);

        TSimpleKeyRing::TKeys keys;
        for (const auto& jsonPoint : config.SubKeys("")) {
            ui64 keyId;
            TStringBuf keyIdBuf = TStringBuf(jsonPoint).After('/');
            Y_ENSURE(TryIntFromString<10>(keyIdBuf, keyId), "Invalid key id: " << keyIdBuf);

            TString key = NUtils::Base64ToBin(config.As<TString>(jsonPoint));
            Y_ENSURE(key.size() == 32, "Invalid key size: " << key.size());
            Y_ENSURE(keys.emplace(keyId, std::move(key)).second, "Duplicated key for id '" << keyId << "'");
        }

        return keys;
    }

    const TString& TSimpleKeyRing::AddKey(ui64 id, TString&& key, EAddKeyPolicy policy) {
        if (policy == EAddKeyPolicy::Remain) {
            auto it = Keys_.find(id);
            if (it != Keys_.end()) {
                Y_ENSURE(it->second == key, "Key for id '" << id << "' changed");
                return it->second;
            }
        }

        const auto& [it, inserted] = Keys_.emplace(id, std::move(key));
        Y_ENSURE(inserted, "Key for id '" << id << "' already exists");

        return it->second;
    }

    bool TSimpleKeyRing::HasKey(ui64 id) const {
        return Keys_.contains(id);
    }

    const TString& TSimpleKeyRing::GetOptionalKey(ui64 id) const {
        auto it = Keys_.find(id);
        if (it != Keys_.end()) {
            return it->second;
        }
        return EMPTY_KEY;
    }

    const TString& TSimpleKeyRing::GetKey(ui64 id) const {
        const TString& key = GetOptionalKey(id);
        Y_ENSURE(key != EMPTY_KEY, "No key for id '" << id << "'");
        return key;
    }

    TKeyRingWithEpoch::TKeyRingWithEpoch(const TString& keyDirPath)
        : KeyDirPath_(keyDirPath)
    {
    }

    bool TKeyRingWithEpoch::HasKey(ui64 id) const {
        return GetOptionalKey(id) == EMPTY_KEY;
    }

    const TString& TKeyRingWithEpoch::GetOptionalKey(ui64 id) const {
        try {
            return GetKey(id);
        } catch (...) {
            return EMPTY_KEY;
        }
    }

    // https://a.yandex-team.ru/arc/trunk/arcadia/passport/python/core/crypto/key_storage.py?rev=5874757#L17
    const TString& TKeyRingWithEpoch::GetKey(ui64 id) const {
        {
            std::shared_lock lock(Mutex_);
            const TString& key = Cache_.GetOptionalKey(id);
            if (key != EMPTY_KEY) {
                return key;
            }
        }

        Y_ENSURE(KeyDirPath_, "Missing key dir path");

        auto createFilename = [this](ui64 epoch) {
            return NUtils::CreateStr(KeyDirPath_, "/", epoch, ".key");
        };

        TString key = NUtils::ReadFile(createFilename(id));

        if (!NFs::Exists(createFilename(id + 1))) {
            TLog::Error() << "KeyRing: next epoch key does not exist, provide new keys ASAP!";
        }

        std::unique_lock lock(Mutex_);
        return Cache_.AddKey(id, std::move(key), TSimpleKeyRing::EAddKeyPolicy::Remain);
    }
}
