#pragma once

#include <util/generic/string.h>

#include <shared_mutex>
#include <unordered_map>

namespace NPassport::NLbchdb::NCrypto {
    class IKeyRing {
    public:
        static const TString EMPTY_KEY;

        virtual ~IKeyRing() = default;

        virtual bool HasKey(ui64 id) const = 0;
        virtual const TString& GetOptionalKey(ui64 id) const = 0;
        virtual const TString& GetKey(ui64 id) const = 0;
    };
    using TKeyRingPtr = std::unique_ptr<IKeyRing>;

    class TKeyRingHolderWithDefaultKey {
    public:
        TKeyRingHolderWithDefaultKey(TKeyRingPtr keyRing, ui64 defaultKeyID);

        ui64 DefaultKeyID() const {
            return DefaultKeyID_;
        }
        const TString& DefaultKey() const {
            return KeyRing_->GetKey(DefaultKeyID_);
        }
        const IKeyRing& KeyRing() const {
            return *KeyRing_;
        }

    private:
        void SetDefaultKeyID(ui64 keyID);

    private:
        ui64 DefaultKeyID_ = 0;
        TKeyRingPtr KeyRing_;
    };

    class TSimpleKeyRing: public IKeyRing {
    public:
        using TKeys = std::unordered_map<ui64, TString>;

        explicit TSimpleKeyRing(TKeys&& keys = {});
        explicit TSimpleKeyRing(const TString& filePath);

        static TKeys ReadFromFile(const TString& filePath);

        enum class EAddKeyPolicy {
            Insert,
            Remain,
        };
        const TString& AddKey(ui64 id, TString&& key, EAddKeyPolicy policy = EAddKeyPolicy::Insert);

        bool HasKey(ui64 id) const override;
        const TString& GetOptionalKey(ui64 id) const override;
        const TString& GetKey(ui64 id) const override;

    private:
        TKeys Keys_;
    };

    class TKeyRingWithEpoch: public IKeyRing {
    public:
        explicit TKeyRingWithEpoch(const TString& keyDirPath);

        bool HasKey(ui64 id) const override;
        const TString& GetOptionalKey(ui64 id) const override;
        const TString& GetKey(ui64 id) const override;

    private:
        const TString KeyDirPath_;

        mutable std::shared_mutex Mutex_;
        mutable TSimpleKeyRing Cache_;
    };
}
