#include "data_pusher.h"

#include <util/stream/format.h>
#include <util/string/join.h>

namespace NPassport::NLbchdb {
    TDataPusher::TDataPusher(TPusherSettings settings, TLbReaderPoolPtr readerPool)
        : TBase(NLb::NPusher::TBaseSettings{
                    .SignalPrefix = "hbase.dirty", // theese signals are not about hbase only already
                    .ErrorSignalName = "push_failes",
                }, readerPool)
    {
        Unistat.HbaseTimeStats = std::make_shared<NUnistat::TTimeStat>(
            "hbase.dirty.response_time",
            std::move(settings.HbaseTimeStatBounds));
        Unistat.YtTimeStats = std::make_shared<NUnistat::TTimeStat>(
            "yt.response_time",
            NUnistat::TTimeStat::CreateBoundsFromMaxValue(settings.YtMaxTimeSignal));
    }

    TDataPusher::~TDataPusher() {
        Stop();
    }

    void TDataPusher::AddUnistat(NUnistat::TBuilder& builder) {
        AddUnistatBase(builder);

        builder.Add(Unistat.HbaseRequests);
        builder.Add(Unistat.HbaseBatches);
        builder.Add(Unistat.HbaseMutations);
        builder.Add(Unistat.HbaseErrors);
        Unistat.HbaseTimeStats->AddUnistat(builder);

        builder.Add(Unistat.YtRequests);
        builder.Add(Unistat.YtWrittenRows);
        builder.Add(Unistat.YtErrors);
        Unistat.YtTimeStats->AddUnistat(builder);
    }

    TDataPushWorker::TDataPushWorker(const TPusherWorkerSettings& settings, size_t idx, TDataPusher& parent)
        : TBase(settings.Base, idx, parent, [this](auto& d) { return Proc(d); })
        , Hbase_(settings.Hbase ? std::make_unique<NHbase::THBasePool>(*settings.Hbase) : nullptr)
        , HbaseBatchSettings_(settings.HbaseBatch)
        , HbaseTablesPrefix_(settings.HbaseTablesPrefix)
        , YtBatchSettings_(settings.YtBatch)
        , YtClient_(settings.YtFactory->Create())
    {
        HbaseBatchSettings_.LogableIdx = idx;

        TLog::Debug() << "TDataPushWorker was created: " << idx;
    }

    bool TDataPushWorker::Proc(TLbReaderPool::TParsedData& data) {
        const TString cookies = JoinRange(",", data.Traceids.begin(), data.Traceids.end());

        std::optional<NYt::TBatch> ytBatch;
        if (!data.Data.Yt.ByTable.empty()) {
            TErrIncrementer err(GetParent().Unistat.YtErrors);

            ytBatch.emplace(YtBatchSettings_, YtClient_);
            for (auto& [table, builder] : data.Data.Yt.ByTable) {
                ytBatch->Send(
                    table,
                    *builder,
                    [&](NYt::TBatch::TQueryResult& req) {
                        ++GetParent().Unistat.YtRequests;
                        GetParent().Unistat.YtWrittenRows += req.Rows;

                        req.Future.Subscribe(
                            [timeStat = GetParent().Unistat.YtTimeStats, start = req.StartTime](const auto&) {
                                timeStat->Insert(TInstant::Now() - start);
                            });
                    });
            }

            err.Clear();
        }

        if (Hbase_ && !data.Data.Hbase.ByTable.empty()) {
            TErrIncrementer err(GetParent().Unistat.HbaseErrors);
            const TInstant start = TInstant::Now();

            NHbase::TBatch hbaseBatch(*Hbase_,
                                      HbaseBatchSettings_,
                                      GetParent().Unistat.HbaseTimeStats,
                                      cookies);

            for (const auto& pair : data.Data.Hbase.ByTable) {
                const std::string& tableName = pair.first;

                if (HbaseTablesPrefix_.empty()) {
                    hbaseBatch.Add(tableName, pair.second);
                } else {
                    hbaseBatch.Add(HbaseTablesPrefix_ + tableName, pair.second);
                }
            }

            GetParent().Unistat.HbaseRequests += hbaseBatch.RequestCount();
            GetParent().Unistat.HbaseBatches += hbaseBatch.BatchCount();
            GetParent().Unistat.HbaseMutations += hbaseBatch.MutationsCount();

            TLog::Debug() << "worker#" << Idx_ << ". Hbase: Succeed to write "
                          << hbaseBatch.BatchCount() << " batches with "
                          << hbaseBatch.MutationsCount() << " mutations "
                          << "in " << hbaseBatch.RequestCount() << " requests. Took: "
                          << HumanReadable(TInstant::Now() - start)
                          << ". cookies: " << cookies;

            err.Clear();
        }

        if (ytBatch) {
            TErrIncrementer err(GetParent().Unistat.YtErrors);

            const bool result = ytBatch->WaitResult(
                [&](const NYt::TBatch::TQueryResult& req) {
                    TLog::Debug() << "worker#" << Idx_
                                  << ". Yt: Trying to write " << req.Rows << " rows to " << req.TableName
                                  << ". Result: " << req.Future.GetValue().Msg
                                  << ". Took (dirty): " << HumanReadable(TInstant::Now() - req.StartTime)
                                  << ". Cookies: " << cookies;
                    return req.Future.GetValue().Status == NYt::TYtClient::TWriteResult::Ok;
                });
            if (!result) {
                return false;
            }

            err.Clear();
        }

        for (NThreading::TPromise<void>& p : data.Promises) {
            p.SetValue();
        }

        return true;
    }

    TErrIncrementer::TErrIncrementer(NUnistat::TSignalDiff<>& sig)
        : Sig_(&sig)
    {
    }

    TErrIncrementer::~TErrIncrementer() {
        if (Sig_) {
            ++(*Sig_);
        }
    }
}
