#include "sender_delivery_extender.h"

namespace NPassport::NLbchdb::NExtend {
    TSenderDeliveryExtender::TSenderDeliveryExtender(IBlackboxClient& bb,
                                                     NUnistat::TSignalDiff<>& bbErrors,
                                                     const TSenderDeliveryExtenderSettings& settings)
        : Bb_(bb)
        , BbErrors_(bbErrors)
        , Settings_(settings)
    {
        Y_ENSURE(Settings_.InflightReqsLimit > 1);
    }

    void TSenderDeliveryExtender::Reserve(size_t size) {
        Result_.reserve(size);
    }

    // copy-paste from https://a.yandex-team.ru/arc/trunk/arcadia/passport/infra/daemons/blackbox/src/misc/utils.cpp?rev=r8919378#L35
    static bool IsValidLogin(const TStringBuf login) {
        auto it = std::find_if(login.cbegin(), login.cend(), [](const char c) {
            return unsigned(c) < ' ';
        });
        return it == login.cend();
    }

    void TSenderDeliveryExtender::Add(const NParser::TSenderDeliveryRow& row) {
        if (row.RecipientEmail.size() > 10 * 1024) {
            // PASSP-35226: user input may be huge
            TLog::Debug() << "SenderDeliveryExtender: skip too long email: '" << row.RecipientEmail << "'";
            return;
        }
        if (!IsValidLogin(row.RecipientEmail)) {
            // PASSPORTDUTY-455: user input may contain invalid chars
            TLog::Debug() << "SenderDeliveryExtender: skip invalid email: '" << row.RecipientEmail << "'";
            return;
        }

        auto addImpl = [this, &row](std::deque<TRequest>& reqs, const IBlackboxClient::TRequest& req) {
            while (Userinfo_.size() + EmailBindings_.size() >= Settings_.InflightReqsLimit) {
                WaitSomeResponses();
            }

            while (!TryAdd(row.UnsubscribeList, row.Ts, reqs, req)) {
            }
        };

        addImpl(Userinfo_, TBlackboxClient::CreateUserinfoRequest(row.RecipientEmail));
        addImpl(EmailBindings_, TBlackboxClient::CreateEmailBindingsRequest(row.RecipientEmail));
    }

    TSenderDeliveryExtender::TResult TSenderDeliveryExtender::Finish() {
        while (!Userinfo_.empty() || !EmailBindings_.empty()) {
            WaitSomeResponses();
        }

        return std::move(Result_);
    }

    bool TSenderDeliveryExtender::TryAdd(const ui64 unsubscribeList,
                                         const time_t ts,
                                         std::deque<TSenderDeliveryExtender::TRequest>& reqs,
                                         const TBlackboxClient::TRequest& req,
                                         EWaitResponses wait) {
        try {
            TBlackboxClient::TRequestHandle r = Bb_.SendRequest(req);

            reqs.push_back(TRequest{
                .UnsubscribeList = unsubscribeList,
                .Ts = ts,
                .BbReq = std::move(r),
            });

            if (Backoff_) {
                Backoff_->Decrease();
            }
            return true;
        } catch (const NDbPool::TCantGetConnection& e) {
            TLog::Debug() << "SenderDeliveryExtender: there are no workers for blackbox: "
                          << e.what();
            if (wait == EWaitResponses::True) {
                WaitSomeResponses();
            }
        } catch (const std::exception& e) {
            TLog::Debug() << "SenderDeliveryExtender: failed to send request to blackbox: "
                          << e.what();
            GetBackoff().Sleep();
            GetBackoff().Increase();
        }

        return false;
    }

    void TSenderDeliveryExtender::WaitSomeResponses() {
        if (Userinfo_.empty() && EmailBindings_.empty()) {
            Sleep(Settings_.SleepOnFailMin);
            return;
        }

        std::optional<TRequest> userinfoToResend = TryWait(
            Userinfo_,
            [&](const TString& resp, TRequest& req) {
                std::optional<ui64> uid = TBlackboxClient::ParseResponseForUserinfo(resp);
                if (uid) {
                    AddResult(*uid, req.UnsubscribeList, req.Ts);
                }
            });

        std::optional<TRequest> emailBindingsToResend = TryWait(
            EmailBindings_,
            [&](const TString& resp, TRequest& req) {
                for (ui64 uid : TBlackboxClient::ParseResponseForEmailBindings(resp)) {
                    AddResult(uid, req.UnsubscribeList, req.Ts);
                }
            });

        // Resend only after processing both containers
        // to avoid live lock: when all workers are busy in second container
        auto resend = [&](std::deque<TRequest>& reqs, std::optional<TRequest>& req) {
            if (req) {
                while (!TryAdd(
                    req->UnsubscribeList,
                    req->Ts,
                    reqs,
                    req->BbReq.Req,
                    EWaitResponses::False /* do not try to wait - to avoid recursion */)) {
                }
            }
        };
        resend(Userinfo_, userinfoToResend);
        resend(EmailBindings_, emailBindingsToResend);
    }

    void TSenderDeliveryExtender::AddResult(ui64 uid, ui64 unsubscribeList, ui64 ts) {
        Result_.push_back(TSenderDeliveryExtendedEntry{
            .Uid = uid,
            .UnsubscribeList = unsubscribeList,
            .Ts = ts,
        });
    }

    NUtils::TExponentialBackoff& TSenderDeliveryExtender::GetBackoff() {
        if (!Backoff_) {
            Backoff_.emplace(Settings_.SleepOnFailMin, Settings_.SleepOnFailMax, 2, .1);
        }
        return *Backoff_;
    }

    bool TSenderDeliveryExtender::TRequest::operator==(const TSenderDeliveryExtender::TRequest& o) const {
        return UnsubscribeList == o.UnsubscribeList &&
               Ts == o.Ts &&
               BbReq == o.BbReq;
    }
}
