#pragma once

#include "utils.h"

#include <passport/infra/daemons/lbchdb/src/blackbox_client.h>
#include <passport/infra/daemons/lbchdb/src/parsers/sender_delivery_parser.h>

#include <passport/infra/libs/cpp/utils/exponential_backoff.h>

#include <deque>

namespace NPassport::NLbchdb::NExtend {
    struct TSenderDeliveryExtendedEntry {
        ui64 Uid = 0;
        ui64 UnsubscribeList = 0;
        ui64 Ts = 0;

        bool operator==(const TSenderDeliveryExtendedEntry& o) const {
            return Uid == o.Uid &&
                   UnsubscribeList == o.UnsubscribeList &&
                   Ts == o.Ts;
        }
    };

    struct TSenderDeliveryExtenderSettings {
        size_t InflightReqsLimit = 10;
        TDuration SleepOnFailMin = TDuration::Seconds(1);
        TDuration SleepOnFailMax = TDuration::Seconds(5);
    };

    class TSenderDeliveryExtender {
    public:
        using TResult = std::vector<TSenderDeliveryExtendedEntry>;

        TSenderDeliveryExtender(IBlackboxClient& bb,
                                NUnistat::TSignalDiff<>& bbErrors,
                                const TSenderDeliveryExtenderSettings& settings = {});

        void Reserve(size_t size);
        void Add(const NParser::TSenderDeliveryRow& row);
        TResult Finish();

    public:
        struct TRequest {
            ui64 UnsubscribeList = 0;
            time_t Ts = 0;
            TBlackboxClient::TRequestHandle BbReq;

            bool operator==(const TRequest& o) const;
        };

        enum class EWaitResponses {
            True,
            False,
        };

        bool TryAdd(const ui64 unsubscribeList,
                    const time_t ts,
                    std::deque<TRequest>& reqs,
                    const TBlackboxClient::TRequest& req,
                    EWaitResponses wait = EWaitResponses::True);

        void WaitSomeResponses();

        template <typename Func>
        std::optional<TRequest> TryWait(std::deque<TRequest>& reqs, Func proc) const;

        void AddResult(ui64 uid, ui64 unsubscribeList, ui64 ts);
        NUtils::TExponentialBackoff& GetBackoff();

    protected:
        std::deque<TRequest> Userinfo_;
        std::deque<TRequest> EmailBindings_;

    private:
        IBlackboxClient& Bb_;
        NUnistat::TSignalDiff<>& BbErrors_;
        const TSenderDeliveryExtenderSettings Settings_;

        std::vector<TSenderDeliveryExtendedEntry> Result_;

        std::optional<NUtils::TExponentialBackoff> Backoff_;
    };

    template <typename Func>
    std::optional<TSenderDeliveryExtender::TRequest>
    TSenderDeliveryExtender::TryWait(std::deque<TSenderDeliveryExtender::TRequest>& reqs,
                                     Func proc) const {
        if (reqs.empty()) {
            return {};
        }

        std::optional<TRequest> currentRequest(std::move(reqs.front()));
        reqs.pop_front();

        TString resp;
        try {
            resp = Bb_.WaitResponse(currentRequest->BbReq);
            proc(resp, *currentRequest);
            currentRequest.reset();
        } catch (const TBlackboxClient::TParseException& e) {
            ++BbErrors_;
            TLog::Debug() << "SenderDeliveryExtender: failed to parse response from blackbox: "
                          << "'" << resp << "'"
                          << e.what();
        } catch (const std::exception& e) {
            ++BbErrors_;
            TLog::Debug() << "SenderDeliveryExtender: failed to wait response from blackbox: "
                          << e.what();
        }

        return currentRequest;
    }
}
