#include "utils.h"

#include <passport/infra/libs/cpp/utils/ipaddr.h>
#include <passport/infra/libs/cpp/utils/log/global.h>

#include <util/generic/hash.h>
#include <util/string/split.h>

namespace NPassport::NLbchdb::NExtend {
    bool IsRealUserIp(const NUtils::TIpAddr& ip) {
        return !ip.IsLoopback() && !ip.IsPrivate() && !ip.IsMulticast();
    }

    bool IsYandexIp(IIpReg::TIpInfo net) {
        return net.IsYandex && !net.IsUser;
    }

    static const NUtils::TIpAddr ZERO("0.0.0.0");

    std::optional<TIpData> TIpData::Create(const NGeobase::IGeobase& geobase,
                                           const IIpReg& ipreg,
                                           const std::string& ip) {
        TString ipTstr(ip);

        NUtils::TIpAddr addr;
        if (ip.empty() || !addr.Parse(ipTstr)) {
            TLog::Debug() << "Invalid ip: '" << ip << "'";
            return {};
        }

        //'ip.geoid': geoid(ip),
        //'ip.as_list': as_list(ip),
        //'ip.packed': pack_ip(ip),
        //'ip.unpacked': str(ip),
        //'ip.is_real': is_real_user_ip(ip),
        //'ip.is_yandex': is_yandex_ip(ip),

        TIpData res{
            .Packed = addr.ToPackedStringV6(),
            .PackedShortest = addr.ToPackedStringShortest(),
            .Unpacked = ip,
            .IsReal = IsRealUserIp(addr),
            .IsYandex = IsYandexIp(ipreg.GetIpInfo(ipTstr)),
        };

        if (addr != ZERO && !addr.IsLoopback()) {
            res.GeoId = geobase.GetRegionId(res.Unpacked);
            res.As = geobase.GetAs(res.Unpacked);
        }

        return res;
    }

    bool TIpData::operator==(const TIpData& o) const {
        return GeoId == o.GeoId &&
               As == o.As &&
               Packed == o.Packed &&
               PackedShortest == o.PackedShortest &&
               Unpacked == o.Unpacked &&
               IsReal == o.IsReal &&
               IsYandex == o.IsYandex;
    }

    static const std::string BROWSER_NAME = "BrowserName";
    static const std::string BROWSER_VERSION = "BrowserVersion";
    static const std::string OS_NAME = "OSName";
    static const std::string OS_FAMILY = "OSFamily";
    static const std::string OS_VERSION = "OSVersion";

    std::optional<TUserAgentData> TUserAgentData::Create(const IUaTraits& uatraits, const TString& ua) {
        if (ua.empty()) {
            return {};
        }

        std::optional<IUaTraits::TData> data = uatraits.GetUaData(ua);
        if (!data || data->Data.empty()) {
            return {};
        }

        //'browser.name': info.get('BrowserName'),
        //'browser.version': info.get('BrowserVersion'),
        //'os.name': info.get('OSName'),
        //'os.family': info.get('OSFamily'),
        //'os.version': info.get('OSVersion'),

        auto get = [&](const auto& key) {
            auto it = data->Data.find(key);
            return it == data->Data.end() ? std::string() : it->second;
        };

        return TUserAgentData{
            .BrowserName = get(BROWSER_NAME),
            .BrowserVersion = get(BROWSER_VERSION),
            .OsName = get(OS_NAME),
            .OsFamily = get(OS_FAMILY),
            .OsVersion = get(OS_VERSION),
        };
    }

    bool TUserAgentData::operator==(const TUserAgentData& o) const {
        return BrowserName == o.BrowserName &&
               BrowserVersion == o.BrowserVersion &&
               OsName == o.OsName &&
               OsFamily == o.OsFamily &&
               OsVersion == o.OsVersion;
    }

    static const THashMap<TStringBuf, TComment::EKey> TOKENS = {
        {"tokid", TComment::EKey::TokenId},
        {"clid", TComment::EKey::ClientId},
        {"devid", TComment::EKey::DeviceId},
        {"devnm", TComment::EKey::DeviceName},
        {"scope", TComment::EKey::Scope},
        {"AP", TComment::EKey::ApplicationPassword},
        {"asrc", TComment::EKey::ASrc},
        {"pos", TComment::EKey::Pos},
    };

    TComment TComment::Create(TStringBuf comment) {
        TComment res;

        while (comment) {
            TStringBuf pair = comment.NextTok(';');
            auto it = TOKENS.find(pair.NextTok('='));
            if (it != TOKENS.end()) {
                res.Values_[(size_t)it->second] = pair;
            }
        }

        return res;
    }

    TStringBuf TComment::Get(TComment::EKey key) const {
        size_t idx = (size_t)key;
        Y_VERIFY(idx < (size_t)EKey::Count);
        return Values_[idx];
    }

    TString TComment::PrintDebug() const {
        TStringStream s;

        std::vector<std::pair<TString, EKey>> arr;
        arr.reserve(TOKENS.size());
        for (const auto& pair : TOKENS) {
            arr.emplace_back(pair.first, pair.second);
        }
        std::sort(arr.begin(), arr.end(), [](const auto& l, const auto& r) {
            return l.second < r.second;
        });

        for (const auto& pair : arr) {
            TStringBuf val = Values_[(size_t)pair.second];
            if (!val.empty()) {
                s << "'" << pair.first << "'->'" << Values_[(size_t)pair.second] << "'; ";
            }
        }

        return s.Str();
    }

    bool TComment::operator==(const TComment& o) const {
        return Values_ == o.Values_;
    }

    std::optional<TLaasInfo> TLaasInfo::Create(TStringBuf pos) {
        if (pos.empty()) {
            return {};
        }

        TLaasInfo laasInfo;

        if (!TryFromString(pos.NextTok('='), laasInfo.Latitude) ||
            !TryFromString(pos.NextTok('='), laasInfo.Longitude) ||
            !TryFromString(pos.NextTok('='), laasInfo.Accuracy))
        {
            TLog::Debug() << "Invalid laas tokens: " << pos;
            return {};
        }

        if (pos.Contains('=')) {
            TLog::Debug() << "Invalid number of tokens for laas info: " << pos;
            return {};
        }

        laasInfo.Precision = pos;

        return laasInfo;
    }

    TString TLaasInfo::ToString() const {
        return NUtils::CreateStr(FloatToString(Latitude), ":", FloatToString(Longitude),
                                 ":", Accuracy, ":", Precision);
    }

    bool TLaasInfo::operator==(const TLaasInfo& o) const {
        return Latitude == o.Latitude &&
               Longitude == o.Longitude &&
               Accuracy == o.Accuracy &&
               Precision == o.Precision;
    }
}
