#include "auths.h"

#include <passport/infra/daemons/lbchdb/src/hbase_converters/utils.h>

#include <passport/infra/daemons/lbchdb/src/helpers/auths_helper.h>

#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/stream/format.h>

namespace NPassport::NLbchdb::NHBaseConv::NAuth {
    const std::string TAuthQuery::TABLE_NAME = "auths";

    /// table 'auths' and 'failed_auth': family: "c"
    /// columns already prepared for params
    static const std::string AUTHS_COLUMN_HOST_ID = "c:host_id";
    static const std::string AUTHS_COLUMN_CLIENT_NAME = "c:client_name";
    static const std::string AUTHS_COLUMN_LOGIN = "c:login";
    static const std::string AUTHS_COLUMN_SID = "c:sid";
    static const std::string AUTHS_COLUMN_TYPE = "c:type";
    static const std::string AUTHS_COLUMN_STATUS = "c:status";
    static const std::string AUTHS_COLUMN_YANDEXUID = "c:yandexuid";
    static const std::string AUTHS_COLUMN_COMMENT = "c:comment";
    // IP
    static const std::string AUTHS_COLUMN_USER_IP = "c:user_ip";
    static const std::string AUTHS_COLUMN_IP_GEOID = "c:ip.geoid";
    static const std::string AUTHS_COLUMN_IP_AS_LIST = "c:ip.as_list";
    static const std::string AUTHS_COLUMN_IP_IS_YANDEX = "c:ip.is_yandex";
    // UA
    static const std::string AUTHS_COLUMN_BROWSER_NAME = "c:browser.name";
    static const std::string AUTHS_COLUMN_BROWSER_VERSION = "c:browser.version";
    static const std::string AUTHS_COLUMN_OS_NAME = "c:os.name";
    static const std::string AUTHS_COLUMN_OS_FAMILY = "c:os.family";
    static const std::string AUTHS_COLUMN_OS_VERSION = "c:os.version";

    NHbase::TQuery TAuthQuery::BuildQuery(const NExtend::TAuthExtendedEntry& entry,
                                          bool isSamplingEnabled) {
        NHbase::TQuery res;
        res.Type = NHbase::TQuery::EType::Put;
        res.Row = TAuthQuery::BuildRow(entry, isSamplingEnabled);

        BuildParams(entry, res);

        return res;
    }

    TString TAuthQuery::BuildRow(const NExtend::TAuthExtendedEntry& entry, bool isSamplingEnabled) {
        // https://a.yandex-team.ru/arc/trunk/arcadia/passport/python/library/historydbloader/historydb/hbase_converter.py?rev=5699294#L162
        TInstant timestamp = entry.Row.Timestamp;

        TString res = NUtils::CreateStrExt(
            128,
            // https://a.yandex-team.ru/arc/trunk/arcadia/passport/python/library/historydbloader/historydb/hbase_converter.py?rev=5699294#L118
            // build_base_auth_key
            BuildKeyEpoch(timestamp.TimeT()),
            "_",
            entry.Row.Uid,
            "_",
            BuildReverseTimestamp(timestamp),
            //
            "_");

        NHelpers::TAuthsHelper::BuildKeyUniquePart(entry, isSamplingEnabled, res);

        return res;
    }

    time_t TAuthQuery::BuildKeyEpoch(time_t timestamp) noexcept {
        // https://a.yandex-team.ru/arc/trunk/arcadia/passport/python/library/historydbloader/historydb/hbase_converter.py?rev=5699294#L102
        constexpr time_t SECONDS_IN_30_DAYS = 60 * 60 * 24 * 30;
        return timestamp / SECONDS_IN_30_DAYS;
    }

    TString TAuthQuery::BuildReverseTimestamp(TInstant timestamp) {
        // https://a.yandex-team.ru/arc/trunk/arcadia/passport/python/library/historydbloader/historydb/hbase_converter.py?rev=5699294#L88
        constexpr ui64 EVENTS_KEY_TIMESTAMP_MS_LIMIT = 1000000;

        ui64 left = std::numeric_limits<i64>::max() - timestamp.Seconds();
        ui64 right = (EVENTS_KEY_TIMESTAMP_MS_LIMIT - timestamp.MicroSecondsOfSecond()) % EVENTS_KEY_TIMESTAMP_MS_LIMIT;

        TStringStream res;
        res.Reserve(20);

        res << LeftPad(left, 19, '0') << "." << LeftPad(right, 6, '0');

        return res.Str();
    }

    void TAuthQuery::BuildParams(const NExtend::TAuthExtendedEntry& entry, NHbase::TQuery& q) {
        // https://a.yandex-team.ru/arc_vcs/passport/python/library/historydbloader/hbase_converter.py?rev=r8069857#L403

        q.AddParam(AUTHS_COLUMN_HOST_ID, std::to_string(entry.Row.HostId));
        q.AddParam(AUTHS_COLUMN_CLIENT_NAME, entry.Row.ClientName);
        q.AddParam(AUTHS_COLUMN_LOGIN, entry.Row.Login);
        q.AddParam(AUTHS_COLUMN_SID, entry.Row.Sid);
        q.AddParam(AUTHS_COLUMN_TYPE, entry.Row.AuthType);
        q.AddParam(AUTHS_COLUMN_STATUS, entry.Row.Status);
        q.AddParam(AUTHS_COLUMN_YANDEXUID, entry.Row.YandexUid);
        q.AddParam(AUTHS_COLUMN_COMMENT, entry.Row.Comment);
        q.AddParam(AUTHS_COLUMN_USER_IP, NExtend::TIpData::ChooseUserIp(entry.Row));
        if (entry.IpData) {
            const NExtend::TIpData& data = *entry.IpData;
            if (data.GeoId) {
                q.AddParam(AUTHS_COLUMN_IP_GEOID, std::to_string(*data.GeoId));
            }
            if (data.As) {
                q.AddParam(AUTHS_COLUMN_IP_AS_LIST, *data.As);
            }
            if (data.IsYandex) {
                q.AddParam(AUTHS_COLUMN_IP_IS_YANDEX, TUtils::AsBool(true));
            }
        }
        if (entry.UaData) {
            const NExtend::TUserAgentData& data = *entry.UaData;
            q.AddParam(AUTHS_COLUMN_BROWSER_NAME, data.BrowserName);
            q.AddParam(AUTHS_COLUMN_BROWSER_VERSION, data.BrowserVersion);
            q.AddParam(AUTHS_COLUMN_OS_NAME, data.OsName);
            q.AddParam(AUTHS_COLUMN_OS_FAMILY, data.OsFamily);
            q.AddParam(AUTHS_COLUMN_OS_VERSION, data.OsVersion);
        }
    }
}
