#include "succed_auths.h"

#include "last_auth.h"

#include <passport/infra/daemons/lbchdb/src/hbase_converters/aggregate_auth_helper.h>
#include <passport/infra/daemons/lbchdb/src/hbase_converters/utils.h>

#include <passport/infra/daemons/lbchdb/src/helpers/auths_helper.h>
#include <passport/infra/daemons/lbchdb/src/proto/auths.pb.h>

#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <library/cpp/string_utils/quote/quote.h>

#include <util/stream/str.h>
#include <util/string/builder.h>
#include <util/string/cast.h>

namespace NPassport::NLbchdb::NHBaseConv::NAuth {
    const std::string TSuccessAuth::TABLE_NAME = "success_auths_uids_by_hour";

    static const std::string SUCCESS_AUTHS_UIDS_BY_HOUR_FAMILY = "i";

    std::optional<NHbase::TQuery> TSuccessAuth::BuildQuery(const NExtend::TAuthExtendedEntry& entry) {
        // https://a.yandex-team.ru/arc/trunk/arcadia/passport/python/library/historydbloader/historydb/hbase_converter.py?rev=5699294#L646
        if (!TAggregateAuthHelper::IsEntryWriteable(entry.IpData) || ShouldSkip(entry)) {
            return {};
        }

        AuthInfo proto;
        proto.set_type(TString(entry.Row.AuthType));

        Y_VERIFY(entry.IpData);
        TAggregateAuthHelper::FillIpInfo(*entry.IpData, proto);

        if (entry.Row.AuthType != "oauthcheck") {
            FillBrowserInfo(entry, proto);
            FillOsInfo(entry, proto);
        }
        FillTokenInfo(entry.Comment, proto);

        TString value = NUtils::CreateStr("1:", TUtils::PrintTimeInPythonStyle(entry.Row.Timestamp));
        if (entry.Laas) {
            NUtils::Append(value, ":", entry.Laas->ToString());
        }

        NHbase::TQuery res;
        res.Type = NHbase::TQuery::EType::Put;
        Y_VERIFY(entry.IpData);
        res.Row = NUtils::CreateStr(
            TAggregateAuthHelper::BuildRow(entry.Row.Uid, entry.Row.Timestamp),
            entry.IpData->PackedShortest,
            ".",
            entry.Row.AuthType);

        const TString protoSerialized = proto.SerializeAsString();

        std::string key;
        key.reserve(SUCCESS_AUTHS_UIDS_BY_HOUR_FAMILY.size() + 1 + protoSerialized.size());
        key.append(SUCCESS_AUTHS_UIDS_BY_HOUR_FAMILY);
        key.append(":");
        key.append(protoSerialized.begin(), protoSerialized.end());

        res.AddParam(std::move(key), TStringBuf(value));

        return res;
    }

    bool TSuccessAuth::ShouldSkip(const NExtend::TAuthExtendedEntry& entry) {
        // Только успешные вебовские и просто успешные (login, oauthcheck)
        if ((entry.Row.Status != "ses_create" && entry.Row.Status != "successful") ||
            !entry.Row.AuthType ||
            entry.Row.AuthType == "unknown" ||
            entry.Row.AuthType == "oauthcreate") {
            return true;
        }

        // Такие события пишутся при любом подтверждении действия паролем
        if (NHelpers::TAuthsHelper::IsWebSuccessful(entry.Row)) {
            return true;
        }

        // Авторизации по xtoken выкидываем из агрегированной истории
        // https://a.yandex-team.ru/arc_vcs/passport/python/library/historydbloader/hbase_converter.py?rev=r8069857#L732
        return entry.Comment.Get(NExtend::TComment::EKey::ASrc) == "xtoken";
    }

    void TSuccessAuth::FillBrowserInfo(const NExtend::TAuthExtendedEntry& entry, AuthInfo& proto) {
        if (!entry.UaData) {
            return;
        }

        const NExtend::TUserAgentData& data = *entry.UaData;

        if (!data.BrowserName.empty()) {
            proto.mutable_browser_info()->set_name((TString)data.BrowserName);
        }
        if (!data.BrowserVersion.empty()) {
            proto.mutable_browser_info()->set_version((TString)data.BrowserVersion);
        }
    }

    void TSuccessAuth::FillOsInfo(const NExtend::TAuthExtendedEntry& entry, AuthInfo& proto) {
        if (!entry.UaData) {
            return;
        }

        const NExtend::TUserAgentData& data = *entry.UaData;

        if (!data.OsName.empty()) {
            proto.mutable_os_info()->set_name((TString)data.OsName);
        }
        if (!data.OsVersion.empty()) {
            proto.mutable_os_info()->set_version((TString)data.OsVersion);
        }
    }

    void TSuccessAuth::FillTokenInfo(const NExtend::TComment& comment, AuthInfo& proto) {
        TStringBuf tokenid = comment.Get(NExtend::TComment::EKey::TokenId);
        TStringBuf clientid = comment.Get(NExtend::TComment::EKey::ClientId);
        if (tokenid.empty() || clientid.empty()) {
            return;
        }

        TokenInfo* info = proto.mutable_token_info();
        info->set_token_id(tokenid.data(), tokenid.size());
        info->set_client_id(clientid.data(), clientid.size());

        TStringBuf deviceid = comment.Get(NExtend::TComment::EKey::DeviceId);
        if (deviceid) {
            info->set_device_id(UrlUnescapeRet(deviceid));
        }

        TStringBuf devicename = comment.Get(NExtend::TComment::EKey::DeviceName);
        if (devicename) {
            info->set_device_name(UrlUnescapeRet(devicename));
        }

        TStringBuf scopes = comment.Get(NExtend::TComment::EKey::Scope);
        if (scopes) {
            info->set_scopes(scopes.data(), scopes.size());
        }

        TStringBuf AP = comment.Get(NExtend::TComment::EKey::ApplicationPassword);
        if (AP == "1") {
            info->set_ap(true);
        }
    }
}
