#include "registration.h"

#include <algorithm>

namespace NPassport::NLbchdb::NHBaseConv::NEvent {
    static const TVector<TStringBuf> REGISTRATION_PREFIXES = {"account_create", "account_register"};

    const TStringBuf TRegistration::TABLE_NAME = "ip_events";

    static TString PackIp(TStringBuf ip, ui32 action) {
        // https://a.yandex-team.ru/arc/trunk/arcadia/passport/python/library/historydbloader/hbase_converter.py?rev=r8070792#L252
        TString binaryIp = TString(ip);
        binaryIp.ReserveAndResize(20);

        NUtils::ToBytesMsb(action, binaryIp, 16);
        return binaryIp;
    }

    static TString PackUniquePart(ui64 uid, ui32 hostId) {
        // https://a.yandex-team.ru/arc/trunk/arcadia/passport/python/library/historydbloader/hbase_converter.py?rev=r8069857#L256
        TString binaryUniquePart(12, '\x00');
        NUtils::ToBytesMsb(uid, binaryUniquePart);
        NUtils::ToBytesMsb(hostId, binaryUniquePart, 8);
        return binaryUniquePart;
    }

    std::optional<NHbase::TQuery> TRegistration::BuildQuery(const NExtend::TEventExtendedEntry& entry) {
        if (entry.Row.Name != "action") {
            return {};
        }

        auto it = std::find_if(REGISTRATION_PREFIXES.begin(), REGISTRATION_PREFIXES.end(), [&](TStringBuf prefix) {
            return entry.Row.EventValue.StartsWith(prefix);
        });

        if (it == REGISTRATION_PREFIXES.end() || !entry.IpData) {
            return {};
        }

        NHbase::TQuery query;

        query.Row = BuildKeyUniquePart(entry);
        PopulateOptionalFields(entry, query);
        query.AddParam(COLUMN_UID, TStringBuf(IntToString<10>(entry.Row.Uid)));

        return query;
    }

    TString TRegistration::BuildKeyUniquePart(const NExtend::TEventExtendedEntry& entry) {
        return NUtils::CreateStr(PackIp(entry.IpData->Packed, 1), BuildReverseTs(entry.Row.Timestamp), PackUniquePart(entry.Row.Uid, entry.Row.HostId.value_or(0)));
    }

    TString TRegistration::BuildReverseTs(TInstant timestamp) {
        TString ts(12, '\x00');
        NUtils::ToBytesMsb(std::numeric_limits<i64>::max() - timestamp.Seconds(), ts);
        NUtils::ToBytesMsb(EVENTS_KEY_TIMESTAMP_MS_LIMIT - ui32(timestamp.MicroSecondsOfSecond()), ts, 8);
        return ts;
    }
}
