#include "auths_helper.h"

#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NLbchdb::NHelpers {
    bool TAuthsHelper::IsSuccessAuth(const NParser::TAuthRow& row, bool includeSesUpdate) {
        // https://a.yandex-team.ru/arc/trunk/arcadia/passport/python/library/historydbloader/historydb/hbase_converter.py?rev=5699294#L515
        // is_success_auth
        if (IsWebSuccessful(row)) {
            return false;
        }

        return row.Status == "ses_create" ||
               row.Status == "successful" ||
               (includeSesUpdate && row.Status == "ses_update");
    }

    bool TAuthsHelper::IsWebSuccessful(const NParser::TAuthRow& row) {
        // Blackbox пишет successful даже если у пользователя
        // установлен 3 бит на 8 сиде
        return row.ClientName == "bb" &&
               row.AuthType == "web" &&
               row.Status == "successful";
    }

    // https://a.yandex-team.ru/arc/trunk/arcadia/passport/python/library/historydbloader/historydb/hbase_converter.py?rev=5699294#L65
    static const THashSet<TString> AUTH_STATUSES_FOR_FAILED_AUTHS_TABLE = {
        "failed",
        "bruteforce",
        "disabled",
        "blocked",
        "challenge_shown", // Это не failed-авторизация, но кладем в таблицу failed_auths
    };

    bool TAuthsHelper::IsFailedAuthStatus(const TStringBuf status) {
        return AUTH_STATUSES_FOR_FAILED_AUTHS_TABLE.contains(status);
    }

    bool TAuthsHelper::IsSampledAuthtype(TStringBuf authtype) {
        return authtype == "oauthcheck";
    }

    TString TAuthsHelper::BuildSamplingKey(const NExtend::TAuthExtendedEntry& entry, ui32 samplingPeriod) {
        Y_VERIFY(samplingPeriod);

        TInstant timestamp = entry.Row.Timestamp;
        timestamp = TInstant::Seconds(timestamp.Seconds());
        timestamp -= TDuration::Seconds(timestamp.Seconds() % samplingPeriod);

        return NUtils::CreateStr(
            timestamp.Seconds(),
            ";",
            entry.Row.Uid,
            ";",
            entry.Comment.Get(NExtend::TComment::EKey::TokenId),
            ";",
            NExtend::TIpData::ChooseUserIp(entry.Row));
    }

    TString TAuthsHelper::BuildKeyUniquePart(const NExtend::TAuthExtendedEntry& entry, bool isSamplingEnabled) {
        TString res;
        BuildKeyUniquePart(entry, isSamplingEnabled, res);
        return res;
    }

    void TAuthsHelper::BuildKeyUniquePart(const NExtend::TAuthExtendedEntry& entry, bool isSamplingEnabled, TString& out) {
        // https://a.yandex-team.ru/arc/trunk/arcadia/passport/python/library/historydbloader/historydb/hbase_converter.py?rev=5699294#L106
        // build_auth_key_unique_part
        if (isSamplingEnabled) {
            TStringBuf tokenId = entry.Comment.Get(NExtend::TComment::EKey::TokenId);
            NUtils::Append(out, entry.Row.AuthType, "/", tokenId ? tokenId : "-");
        } else {
            NUtils::Append(out, entry.Row.HostId);
        }
    }
}
