#include "lbchdb.h"

#include "sender_unsubsrcibe_lists.h"
#include "extenders/sender_delivery_extender.h"
#include "parsers/auth_parser.h"
#include "yt_converters/auth/auths.h"
#include "yt_converters/mail_user_journal/users_history.h"
#include "yt_converters/push/push.h"
#include "yt_converters/push/push_by_app_id.h"
#include "yt_converters/push/push_by_device_id.h"
#include "yt_converters/push/push_by_uid.h"

#include <passport/infra/libs/cpp/dbpool/db_pool_ctx.h>
#include <passport/infra/libs/cpp/dbpool/handle.h>
#include <passport/infra/libs/cpp/hbase/hbase_pool.h>
#include <passport/infra/libs/cpp/json/config.h>
#include <passport/infra/libs/cpp/juggler/status.h>
#include <passport/infra/libs/cpp/tvm/dbpool/create_service_ticket_opt.h>
#include <passport/infra/libs/cpp/tvm/logger/logger.h>
#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/format.h>

#include <library/cpp/geobase/lookup.hpp>

#include <util/stream/file.h>
#include <util/stream/format.h>
#include <util/string/subst.h>
#include <util/system/env.h>

namespace NPassport::NLbchdb {
    TLbchdb::TLbchdb()
        : DbCtx_(NDbPool::TDbPoolCtx::Create())
    {
    }

    TLbchdb::~TLbchdb() {
        Stop();
        TLog::Debug() << "TLbchdb was destroyed";
    }

    void TLbchdb::Init(const NJson::TConfig& config) {
        const TString path = "/component";

        InitLoggers(config, path);

        SetEnv("TZ", "Europe/Moscow");
        tzset();

        InitTvm(config, path + "/tvm");
        InitGeobase(config, path + "/geobase");
        InitUaTraits(config, path + "/uatraits");
        InitLogbroker(config, path + "/logbroker");
        TPusherWorkerSettings settings;
        InitHbase(config, path + "/hbase", settings);
        InitYt(config, path + "/yt", settings);
        InitBlackbox(config, path + "/blackbox");
        InitKolmogor(config, path + "/kolmogor");
        InitDataPusher(config, path + "/data_pusher", settings);
        InitProcessor(config, path + "/processor", path + "/yt/create_tables");

        Start();
    }

    void TLbchdb::HandleRequest(NCommon::TRequest& req) {
        TStringBuf path = req.GetPath();
        path.ChopSuffix("/");

        if (path == "/healthcheck") {
            HandleHealthCheck(req);
            return;
        }

        TLog::Warning() << "Unknown path: " << path;
        req.SetStatus(HTTP_NOT_FOUND);
        req.Write(TStringBuilder() << "Unknown path: " << path);
    }

    void TLbchdb::AddUnistat(NUnistat::TBuilder& builder) {
        Processor_->AddUnistat(builder);
        DataPusher_->AddUnistat(builder);
        Lb_->AddUnistat(builder);
        UncommittedMemoryDispatcher_->AddUnistat(builder);
        DbCtx_->AddUnistat(builder);
    }

    void TLbchdb::RotateLogs() {
        YtFactory_->RotateLogs();
    }

    void TLbchdb::InitLoggers(const NJson::TConfig& config, const TString& path) {
        config.InitCommonLog(path + "/logger");
        DbCtx_->InitLogger(config.As<TString>(path + "/dbpool_log"));
    }

    void TLbchdb::InitTvm(const NJson::TConfig& config, const TString& path) {
        const TString secret = NUtils::ReadFile(config.As<TString>(path + "/self_secret_file"));

        NTvmAuth::NTvmApi::TClientSettings::TDstMap dsts;
        for (const TString& p : config.SubKeys(path + "/dsts")) {
            Y_ENSURE(dsts.emplace(config.As<TString>(p + "/alias"), config.As<ui32>(p + "/tvm_id"))
                         .second,
                     "there are duplicated key: " << p);
        }

        NTvmAuth::NTvmApi::TClientSettings setts;
        setts.SetSelfTvmId(config.As<ui32>(path + "/self_tvm_id"));
        setts.EnableServiceTicketsFetchOptions(secret, std::move(dsts));
        setts.SetDiskCacheDir(config.As<TString>(path + "/disk_cache"));

        setts.TvmHost = config.As<TString>(path + "/tvm_host", setts.TvmHost);
        setts.TvmPort = config.As<ui16>(path + "/tvm_port", setts.TvmPort);

        Tvm_ = std::make_shared<NTvmAuth::TTvmClient>(setts, NTvmLogger::TLogger::Create());

        TLog::Info() << "Tvm inited";
    }

    void TLbchdb::InitGeobase(const NJson::TConfig& config, const TString& path) {
        const TString geoFile = config.As<TString>(path + "/geobase_file");
        Geobase_ = std::make_unique<NGeobase::TGeobase>(geoFile);

        TLog::Info() << "Geobase succeed to load from " << geoFile;

        const TString ipregFile = config.As<TString>(path + "/ipreg_file");
        Ipreg_ = std::make_unique<TIpReg>(ipregFile);

        TLog::Info() << "IpReg succeed to load from " << ipregFile;
    }

    void TLbchdb::InitUaTraits(const NJson::TConfig& config, const TString& path) {
        const TString file = config.As<TString>(path + "/file");
        Uatraits_ = std::make_unique<TUaTraits>(file);

        TLog::Info() << "UaTraits succeed to load from " << file;
    }

    static const TString WEIGHT_CFG_KEY = "/memory_usage_weight";

    static size_t CalcWeightSum(const NJson::TConfig& config, const TString& path) {
        size_t res = 0;

        for (const TString& pathPrefix : config.SubKeys(path + "/logs")) {
            res += config.As<size_t>(pathPrefix + WEIGHT_CFG_KEY);
        }

        TLog::Debug() << "LB: weight_sum=" << res;
        Y_ENSURE(res > 0, "weight_sum cannot be 0");

        return res;
    }

    void TLbchdb::InitLogbroker(const NJson::TConfig& config, const TString& path) {
        Lb_ = std::make_shared<TLbReaderPool>(
            NLb::TReaderPoolSettings{
                .QueueSizeLimit = config.As<ui32>(path + "/queue_size"),
                .Workers = config.As<ui32>(path + "/queue_workers"),
                .JoinMessages = config.As<ui32>(path + "/join_messages", 1),
            });

        NLb::TReaderSettings setts;
        setts.ClientId = config.As<TString>(path + "/client_id");
        setts.ServerName = config.As<TString>(path + "/server");
        if (config.Contains(path + "/port")) { // For tests
            setts.ServerPort = config.As<ui16>(path + "/port");
        } else {
            setts.TvmClient = Tvm_;
        }
        setts.LogLevel = config.As<ui32>(path + "/log_level");
        setts.MaxMemoryUsage = 1024 * 1024 * config.As<size_t>(path + "/memory_usage_per_server__mb");
        setts.ReadMaxBytes = config.As<size_t>(path + "/read_max_bytes");
        setts.ReadMaxCounts = config.As<size_t>(path + "/read_max_count", setts.ReadMaxCounts);
        setts.ReserveRatio = config.As<double>(path + "/reserve_ratio", setts.ReserveRatio);
        setts.InflightReads = config.As<size_t>(path + "/inflight_reads", setts.InflightReads);
        setts.ReadMirroredPartitions = true;

        UncommittedMemoryDispatcher_ = setts.UncommittedMemory;

        const size_t maxUncommittedTotalSize = config.As<size_t>(path + "/max_uncommitted_total_size");
        TLog::Debug() << "LB: max_uncommitted_total_size="
                      << HumanReadableSize(maxUncommittedTotalSize, SF_BYTES);

        const size_t weightSum = CalcWeightSum(config, path);

        for (const TString& pathPrefix : config.SubKeys(path + "/logs")) {
            const TString type = NJson::TConfig::GetKeyFromPath(pathPrefix);

            const size_t weight = config.As<size_t>(pathPrefix + WEIGHT_CFG_KEY);
            const double ratio = double(weight) / weightSum;
            setts.MaxUncommittedSize = ratio * maxUncommittedTotalSize;
            TLog::Debug() << "LB: '" << type
                          << "': max_uncommitted_size=" << HumanReadableSize(setts.MaxUncommittedSize, SF_BYTES)
                          << ". [" << weight << "/" << weightSum
                          << "] ~= " << Prec(100 * ratio, PREC_POINT_DIGITS, 1) << "%";

            setts.TvmDstAlias = setts.ServerName;

            setts.Topics = {config.As<TString>(pathPrefix + "/topic")};
            setts.UnistatSignalId = setts.Topics.front();

            const ELogType enumType = TLogTypeTraits::FromString(type);
            Lb_->AddReader(setts, enumType);

            LogTypes_.push_back(enumType);
        }

        Y_ENSURE(Lb_->GetReadersCount(), "There is no one server in config");

        TLog::Info() << "LB inited";
    }

    void TLbchdb::InitHbase(const NJson::TConfig& config, const TString& path, TPusherWorkerSettings& outSettings) {
        if (!config.As<bool>(path + "/enabled", true)) {
            return;
        }

        NHbase::TSettings hbase;

        ui16 port = config.As<ui32>(path + "/port");
        for (const TString& host : config.As<std::vector<TString>>(path + "/hosts")) {
            hbase.Hosts.emplace_back(NHbase::TSettings::THost{
                .Host = host,
                .Port = port,
            });
        }

        Y_ENSURE(!hbase.Hosts.empty(), "No one hbase host in config");

        auto readTimeout = [&config, path](TDuration& value, const TString& key) {
            value = TDuration::MilliSeconds(config.As<ui32>(path + "/" + key, value.MilliSeconds()));
        };
        readTimeout(hbase.ConnectTimeout, "connect_timeout__ms");
        readTimeout(hbase.SendTimeout, "send_timeout__ms");
        readTimeout(hbase.RecvTimeout, "recv_timeout__ms");
        readTimeout(hbase.ReopenAfterIdle, "reopen_after_idle__ms");

        outSettings.Hbase = std::move(hbase);
        outSettings.HbaseBatch.Limit = config.As<size_t>(path + "/batch_size", 5000);
        TLog::Info() << "Hbase: use batch size=" << outSettings.HbaseBatch.Limit;
        outSettings.HbaseBatch.Retries = config.As<ui32>(path + "/retries_per_request", 20);
        outSettings.HbaseBatch.VerboseLogs = config.As<bool>(path + "/verbose_logs", false);

        outSettings.Base.MinBackoff = TDuration::Seconds(config.As<ui32>(path + "/backoff_min__sec", 1));
        outSettings.Base.MaxBackoff = TDuration::Seconds(config.As<ui32>(path + "/backoff_max__sec", 90));

        outSettings.HbaseTablesPrefix = config.As<TString>(path + "/tables_prefix", "");
    }

    void TLbchdb::InitYt(const NJson::TConfig& config, const TString& path, TPusherWorkerSettings& outSettings) {
        NYt::TYtSettings settings;
        settings.OauthToken = NUtils::ReadFile(config.As<TString>(path + "/oauth_token_path"));
        settings.Cluster = config.As<TString>(path + "/cluster");
        settings.CypressDir = config.As<TString>(path + "/cypress_dir");
        settings.LogFile = config.As<TString>(path + "/log/file");
        settings.LogLevel = config.As<int>(path + "/log/level");

        YtFactory_ = std::make_shared<NYt::TYtClientFactory>(settings);
        TLog::Info() << "YT: configured";

        InitYtTables(config, path + "/create_tables");

        outSettings.YtFactory = YtFactory_;
        outSettings.YtBatch.Limit = config.As<size_t>(path + "/batch_size");
        outSettings.YtBatch.Timeout = TDuration::MilliSeconds(config.As<size_t>(path + "/timeout__ms"));
    }

    static TDuration GetTableTtl(
        const NJson::TConfig& config,
        const TString& path,
        const TString& tableName) {
        const TString tableTtlPath = NUtils::CreateStr(path, '/', tableName, "_table_ttl__sec");
        if (config.Contains(tableTtlPath)) {
            return TDuration::Seconds(config.As<ui64>(tableTtlPath));
        }
        return TDuration::Seconds(config.As<ui64>(NUtils::CreateStr(path, "/default_table_ttl__sec")));
    }

    static NYt::TTableCreatorSettings GetTableCreatorSettings(
        const NJson::TConfig& config,
        const TString& path,
        const TString& tableName,
        NYt::TTableSchema&& schema) {
        TDuration ttl = GetTableTtl(config, path, tableName);

        return NYt::TTableCreatorSettings{
            .CreateSettings = NYt::TYtClient::TCreateSettings{
                .TableTtl = ttl,
                .ErasureCodec = "isa_lrc_12_2_2",
                .DesiredTabletCount = config.As<ui64>(path + "/desired_tablet_count"),
                .StoreChecksum = true,
                .MinDataVersions = 0,
                .MaxDataVersions = 1,
                .MinDataTtl = 0,
                .MaxDataTtl = ttl.MilliSeconds(),
                .Schema = std::move(schema),
                .Timeout = TDuration::Seconds(config.As<ui64>(path + "/create_timeout__sec")),
            },
            .TablePrefix = tableName + '/',
            .PeriodType = NYt::TPeriodicTable::Month,
            .CheckPeriod = TDuration::MilliSeconds(config.As<ui64>(path + "/check_period__msec", 10000)),
        };
    }

    void TLbchdb::InitYtTables(const NJson::TConfig& config, const TString& path) {
        auto addCreator = [this, &config](const TString& path, const TString& tableName, NYt::TTableSchema&& schema) {
            TableCreators_.push_back(std::make_unique<NYt::TTableCreator>(
                YtFactory_, GetTableCreatorSettings(config, path, tableName, std::move(schema))));
        };

        addCreator(
            path + "/auths",
            NYtConv::NAuth::TAuths::AUTHS_TABLE_DIR,
            NYtConv::NAuth::TAuths::CreateYtSchema());

        addCreator(
            path + "/users_history",
            NYtConv::NMailUserJournal::TUsersHistory::TABLE_NAME,
            NYtConv::NMailUserJournal::TUsersHistory::CreateYtSchema());
        addCreator(
            path + "/users_history",
            NYtConv::NMailUserJournal::TUsersHistory::CORP_TABLE_NAME,
            NYtConv::NMailUserJournal::TUsersHistory::CreateYtSchema());

        addCreator(
            path + "/push",
            NYtConv::NPush::TPushTable::TABLE_NAME,
            NYtConv::NPush::TPushTable::CreateYtSchema());
        addCreator(
            path + "/push",
            NYtConv::NPush::TPushByUidTable::TABLE_NAME,
            NYtConv::NPush::TPushByUidTable::CreateYtSchema());
        addCreator(
            path + "/push",
            NYtConv::NPush::TPushByDeviceIdTable::TABLE_NAME,
            NYtConv::NPush::TPushByDeviceIdTable::CreateYtSchema());
        addCreator(
            path + "/push",
            NYtConv::NPush::TPushByAppIdTable::TABLE_NAME,
            NYtConv::NPush::TPushByAppIdTable::CreateYtSchema());
    }

    void TLbchdb::InitBlackbox(const NJson::TConfig& config, const TString& path) {
        NDbPool::TQueryOpts opts = {NTvmDbPool::CreateServiceTicketOpt(
            Tvm_,
            "blackbox")}; // alias from config
        Blackbox_ = DbCtx_->CreateDb(config, path, std::move(opts));
        Blackbox_->TryPing();

        for (IBlackboxClient::TRequest query : {
                 TBlackboxClient::CreateUserinfoRequest("", "/blackbox/check_grants"),
                 TBlackboxClient::CreateEmailBindingsRequest("", "/blackbox/check_grants"),
             }) {
            NDbPool::TQuery q(query.Path);
            q.SetHttpBody(std::move(query.Body));
            const NDbPool::THttpResponse response =
                NDbPool::TBlockingHandle(*Blackbox_).Query(std::move(q))->ToHttpResponse();

            Y_ENSURE(response.Status == 200,
                     "Failed to try blackbox on start: " << response.Status << ": " << response.Body);
        }

        TLog::Info() << "Blackbox: succeed to connect";
    }

    void TLbchdb::InitKolmogor(const NJson::TConfig& config, const TString& path) {
        if (!config.As<bool>(path + "/enabled")) {
            TLog::Info() << "Kolmogor: DISABLED";
            return;
        }

        NDbPool::TQueryOpts opts = {NTvmDbPool::CreateServiceTicketOpt(
            Tvm_,
            "kolmogor")}; // alias from config
        Kolmogor_ = DbCtx_->CreateDb(config, path, std::move(opts));
        Kolmogor_->TryPing();

        TLog::Info() << "Kolmogor: succeed to connect";
    }

    void TLbchdb::PingKolmogorSpace(const TString& space) {
        if (!Kolmogor_) {
            return;
        }

        NDbPool::TBlockingHandle h(*Kolmogor_);
        const NDbPool::THttpResponse response =
            h.Query(NUtils::CreateStr(
                        "/get?space=", NUtils::Urlencode(space),
                        "&keys=qqq"))
                ->ToHttpResponse();
        Y_ENSURE(response.Status == 200,
                 "Failed to try kolmogor on start: " << response.Status << ": " << response.Body);
    }

    void TLbchdb::InitDataPusher(const NJson::TConfig& config,
                                 const TString& path,
                                 const TPusherWorkerSettings& settings) {
        DataPusher_ = std::make_unique<TDataPusher>(
            TPusherSettings{
                .HbaseTimeStatBounds = NUnistat::TTimeStat::CreateBoundsFromMaxValue(TDuration::MilliSeconds(
                    config.As<ui32>(path + "/max_response_time_in_signal__ms", 50000))),
                .YtMaxTimeSignal = settings.YtBatch.Timeout,
            },
            Lb_);

        TLog::Debug() << "TDataPusher created";

        const size_t workers = config.As<ui32>(path + "/workers", 1);
        DataPusher_->CreateWorkers<TDataPushWorker, TPusherWorkerSettings>(
            workers,
            settings);
        TLog::Info() << "DataPusher use " << workers << " workers";
    }

    static const THashMap<NCrypto::TEncryptor::EKeyRing, TString> KEY_RINGS = {
        {NCrypto::TEncryptor::EKeyRing::Restore, "restore"},
        {NCrypto::TEncryptor::EKeyRing::YasmsPrivate, "yasms_private"},
    };

    void TLbchdb::InitProcessor(const NJson::TConfig& config, const TString& path, const TString& createTablesPath) {
        NCrypto::TKeyRingPtr decryptorKeyRing = std::make_unique<NCrypto::TKeyRingWithEpoch>(
            config.As<TString>(path + "/decryptor_key_dir_path"));

        TString encryptorKeyDir = config.As<TString>(path + "/encryptor_key_dir_path");
        NCrypto::TEncryptor::TKeyRingMap encryptorKeyRingMap;
        for (const auto& [ring, alias] : KEY_RINGS) {
            encryptorKeyRingMap.try_emplace(
                ring,
                std::make_unique<NCrypto::TSimpleKeyRing>(encryptorKeyDir + "/" + alias + ".keys"),
                config.As<ui64>(path + "/" + alias + "_default_key"));
        }

        const TString badLinesLog = config.As<TString>(path + "/bad_lines_logfile_pattern");

        std::vector<TString> types;
        for (const ELogType& type : LogTypes_) {
            types.push_back(TLogTypeTraits::ToString(type));
        }

        NLb::TBadLineLoggers<TLogTypeTraits> loggers;
        loggers.Init(badLinesLog, types);

        auto getTableTtl = [&](const TString& path, const TString& tableName = "default") {
            return GetTableTtl(config, createTablesPath + path, tableName);
        };

        NSampler::TSamplerSettings authSamplerSettings{
            .EntryLimit = config.As<size_t>(path + "/auth_sampling/cache_entry_size_limit"),
            .KolmogorSpace = config.As<TString>(path + "/auth_sampling/kolmogor_space"),
        };
        PingKolmogorSpace(authSamplerSettings.KolmogorSpace);

        TProcessorSettings settings{
            .IsLastauthToHbaseEnabled = config.As<bool>(path + "/enable_lastauth_to_hbase"),
            .IsMailToHbaseEnabled = config.As<bool>(path + "/enable_mail_to_hbase"),
            .IsMailToYtEnabled = config.As<bool>(path + "/enable_mail_to_yt"),
            .CompressIfMoreThan = config.As<size_t>(path + "/compress_if_more_than", 100 * 1024),
            .SenderSampleRatio = config.As<uint32_t>(path + "/sender_sample_ratio"),
            .AuthSamplingPeriod = config.As<ui32>(path + "/auth_sampling/period"),
            .AuthSampler = authSamplerSettings,
            .TablesLifeTimeConfig = {
                .MailProdTtl = getTableTtl("/users_history", NYtConv::NMailUserJournal::TUsersHistory::TABLE_NAME),
                .MailCorpTtl = getTableTtl("/users_history", NYtConv::NMailUserJournal::TUsersHistory::CORP_TABLE_NAME),
                .PushTtl = getTableTtl("/push"),
                .AuthsTtl = getTableTtl("/auths", NYtConv::NAuth::TAuths::AUTHS_TABLE_DIR),
                .FutureFrontier = TDuration::Days(1),
            },
            .PushSubscriptionBucketWidth = TDuration::Seconds(config.As<ui64>(path + "/push_subscription_bucket_width")),
        };

        Y_ENSURE(settings.AuthSamplingPeriod > 0, "illegal sampling period");

        std::unique_ptr lists = std::make_unique<TSenderUnsubscribeLists>(
            NUtils::ReadFile(config.As<TString>(path + "/sender_accounts")));

        Processor_ = std::make_unique<TProcessor>(
            *Geobase_,
            *Ipreg_,
            *Uatraits_,
            settings,
            std::move(loggers),
            std::move(decryptorKeyRing),
            std::move(encryptorKeyRingMap),
            std::move(lists),
            Blackbox_.get(),
            Kolmogor_.get());
    }

    static const TString CONTENT_TYPE = "Content-type";
    static const TString CONTENT_TYPE_TEXT = "text/plain";

    void TLbchdb::HandleHealthCheck(NCommon::TRequest& req) {
        NJuggler::TStatus status;
        status.Update(Tvm_->GetStatus());

        TString buf;
        if (Kolmogor_ && !Kolmogor_->IsOk(&buf)) {
            status.Update(NJuggler::ECode::Critical, buf);
        }

        req.Write(status);
        req.SetHeader(CONTENT_TYPE, CONTENT_TYPE_TEXT);
        req.SetStatus(HTTP_OK);
    }

    void TLbchdb::Start() {
        Lb_->Start(
            "lbchdb_",
            [this](const NLb::TDataSet<ELogType>& data) {
                return Processor_->Proc(data);
            });
    }

    void TLbchdb::Stop() {
        TableCreators_.clear();
        Tvm_.reset();

        Lb_.reset(); // stop readers which push this data pipeline

        DataPusher_.reset(); // stop parsers - last users of HBase or YT
        YtFactory_.reset();  // calls NYT::Shutdown() after freeing of all pointers
    }
}
