#include "auth_parser.h"

#include <passport/infra/daemons/lbchdb/src/processor.h>

#include <passport/infra/libs/cpp/utils/string/split.h>

#include <util/generic/set.h>

namespace NPassport::NLbchdb::NParser {
    static TStringBuf ProcIp(TStringBuf val) {
        if (val == "-") {
            return {};
        }

        val.SkipPrefix("::ffff:");
        return val;
    }

    bool TAuthRow::TryParse(NRowParsers::TCsvRow&& row) {
        Row_ = std::move(row);

        if (Row_.GetField(EAuthRow::Uid) == "-") {
            return false;
        }

        Y_ENSURE(Row_.FieldsCount() == (size_t)EAuthRow::COUNT,
                 "unexpected fields count: " << Row_.FieldsCount());
        Y_ENSURE(Row_.GetField(EAuthRow::Version) == "1",
                 "unexpected version: '" << Row_.GetField(EAuthRow::Version) << "'");

        // Got something like: 2010-03-16T11:02:55.123456+03 - without minutes in TZ
        Y_ENSURE(TInstant::TryParseIso8601(Row_.GetField(EAuthRow::Time), Timestamp),
                 "invalid time: '" << Row_.GetField(EAuthRow::Time) << "'");

        Y_ENSURE(TryIntFromString<16>(Row_.GetField(EAuthRow::HostId), HostId),
                 "invalid hostid: '" << Row_.GetField(EAuthRow::HostId) << "'");

        Y_ENSURE(TryIntFromString<10>(Row_.GetField(EAuthRow::Uid), Uid),
                 "invalid uid: '" << Row_.GetField(EAuthRow::Uid) << "'");

        auto getStr = [&](EAuthRow f, TStringBuf& s) {
            if (TStringBuf val = Row_.GetField(f); val != "-") {
                s = Row_.GetField(f);
            }
        };

        auto getAllocatedStr = [&](EAuthRow f, TString& s) {
            if (TStringBuf val = Row_.GetField(f); val != "-") {
                s = Row_.GetAllocatedField(f);
            }
        };

        getStr(EAuthRow::Login, Login);
        {
            TStringBuf sid = Row_.GetField(EAuthRow::Sid);
            ui64 tmp = 0;
            if (TryIntFromString<10>(sid, tmp)) {
                Sid = sid;
            }
        }

        ClientName = Row_.GetField(EAuthRow::ClientName);
        AuthType = Row_.GetField(EAuthRow::Type);
        Status = Row_.GetField(EAuthRow::Status);
        UserIp = ProcIp(Row_.GetField(EAuthRow::UserIp));
        ProxyIp = ProcIp(Row_.GetField(EAuthRow::ProxyIp));
        getStr(EAuthRow::Yandexuid, YandexUid);
        getStr(EAuthRow::Comment, Comment);
        getAllocatedStr(EAuthRow::UserAgent, UserAgent);

        return true;
    }

    bool TAuthRow::operator==(const TAuthRow& o) const {
        return Timestamp == o.Timestamp &&
               HostId == o.HostId &&
               Uid == o.Uid &&
               Login == o.Login &&
               Sid == o.Sid &&
               AuthType == o.AuthType &&
               Status == o.Status &&
               Comment == o.Comment &&
               UserIp == o.UserIp &&
               ProxyIp == o.ProxyIp &&
               YandexUid == o.YandexUid &&
               UserAgent == o.UserAgent;
    }

    TAuthParser::TAuthParser(ELogType logType, NUtils::ILogger* logger)
        : TParserWithBadLineLog(TLogTypeTraits{}, logType, logger)
    {
    }

    TAuthParser::TResult TAuthParser::Parse(const NLb::TData& data) {
        ParseRows(data);
        return std::move(Result_);
    }

    bool TAuthParser::ParseRow(TStringBuf line) {
        std::optional<TAuthRow> row = TryParse(line);
        if (!row) {
            return false;
        }

        Result_.push_back(std::move(*row));
        return true;
    }

    void TAuthParser::Reserve(size_t size) {
        Result_.reserve(Result_.size() + size);
    }

    std::optional<TAuthRow> TAuthParser::TryParse(TStringBuf line) {
        TAuthRow row;

        if (!row.TryParse(line) || row.Uid == 0) {
            return {};
        }

        return row;
    }
}
