#include "event_parser.h"

#include <passport/infra/daemons/lbchdb/src/extenders/utils.h>

namespace NPassport::NLbchdb::NParser {
    static const THashMap<ui8, THashSet<TString>> EVENT_LOG_SENSITIVE_FIELDS_BY_VERSION{
        {1, {}},
        {
            2,
            {
                {"info.password"},
                {"info.hinta"},
            },
        },
    };

    static bool IsFieldEmpty(const NRowParsers::TCsvRow& row, EEventRow field) {
        return row.GetField(field) == "-";
    }

    template <class T>
    static void GetOptionalField(const NRowParsers::TCsvRow& row, EEventRow field, T& value) {
        if (!IsFieldEmpty(row, field)) {
            value = row.GetField(field);
        }
    }

    template <typename T, int base = 10>
    static void GetOptionalField(const NRowParsers::TCsvRow& row, EEventRow field, std::optional<T>& value) {
        static_assert(std::is_integral_v<T>);

        if (!IsFieldEmpty(row, field)) {
            value = 0;
            Y_ENSURE(TryIntFromString<base>(row.GetField(field), *value));
        }
    }

    bool TEventRow::operator==(const TEventRow& o) const {
        return TBaseEventRow::operator==(o) &&
               ClientName == o.ClientName &&
               EventValue == o.EventValue &&
               UserIp == o.UserIp &&
               ProxyIp == o.ProxyIp &&
               YandexUid == o.YandexUid &&
               AdminLogin == o.AdminLogin &&
               Comment == o.Comment;
    }

    TEventParser::TEventParser(const NCrypto::TDecryptor& decryptor,
                               NUtils::ILogger* logger)
        : NLb::TParserWithBadLineLog(TLogTypeTraits{}, ELogType::Event, logger)
        , Decryptor_(decryptor)
    {
    }

    TEventParser::TResult TEventParser::Parse(const NLb::TData& data) {
        ParseRows(data);
        return std::move(Result_);
    }

    bool TEventParser::ParseRow(TStringBuf line) {
        Result_.push_back(TryParse(line));
        return true;
    }

    void TEventParser::Reserve(size_t size) {
        Result_.reserve(Result_.size() + size);
    }

    TEventRow TEventParser::TryParse(TStringBuf line) const {
        NRowParsers::TCsvRow csv(line);
        Y_ENSURE(csv.FieldsCount() == size_t(EEventRow::COUNT), "Invalid field count in entry");

        TEventRow entry;

        ui8 version;
        Y_ENSURE(TryIntFromString<10>(csv.GetField(EEventRow::Version), version), "Invalid version");
        Y_ENSURE(EVENT_LOG_SENSITIVE_FIELDS_BY_VERSION.contains(version), "Invalid version");

        Y_ENSURE(TInstant::TryParseIso8601(csv.GetField(EEventRow::RfcTime), entry.Timestamp), "Invalid rfctime");
        Y_ENSURE(TryIntFromString<10>(csv.GetField(EEventRow::Uid), entry.Uid), "Invalid UID");

        Y_ENSURE(!IsFieldEmpty(csv, EEventRow::EventName), "Event name cannot be empty");
        entry.Name = csv.GetField(EEventRow::EventName);
        if (!IsFieldEmpty(csv, EEventRow::EventValue)) {
            if (EVENT_LOG_SENSITIVE_FIELDS_BY_VERSION.at(version).contains(*entry.Name)) {
                entry.EventValue = Decryptor_.Decrypt(csv.GetField(EEventRow::EventValue));
            } else {
                entry.EventValue = csv.GetField(EEventRow::EventValue);
            }
        }

        GetOptionalField<ui32, 16>(csv, EEventRow::HostId, entry.HostId);
        GetOptionalField(csv, EEventRow::ClientName, entry.ClientName);
        GetOptionalField(csv, EEventRow::UserIp, entry.UserIp);
        GetOptionalField(csv, EEventRow::ProxyIp, entry.ProxyIp);
        GetOptionalField(csv, EEventRow::YandexUid, entry.YandexUid);
        GetOptionalField(csv, EEventRow::AdminLogin, entry.AdminLogin);
        GetOptionalField(csv, EEventRow::Comment, entry.Comment);

        return entry;
    }
}
