#include "oauth_parser.h"

#include <passport/infra/libs/cpp/logbroker/row_parsers/tskv_row.h>

#include <cmath>

namespace NPassport::NLbchdb::NParser {
    bool TOAuthRow::operator==(const TOAuthRow& o) const {
        return Timestamp == o.Timestamp &&
               ClientId == o.ClientId &&
               DeviceId == o.DeviceId &&
               DeviceName == o.DeviceName &&
               Scopes == o.Scopes &&
               UserIp == o.UserIp &&
               Uid == o.Uid &&
               TokenId == o.TokenId;
    }

    TOAuthParser::TOAuthParser(NUtils::ILogger* logger)
        : TParserWithBadLineLog(TLogTypeTraits{}, ELogType::OAuth, logger)
    {
    }

    TOAuthParser::TResult TOAuthParser::Parse(const NLb::TData& data) {
        ParseRows(data);
        return std::move(Result_);
    }

    bool TOAuthParser::ParseRow(TStringBuf line) {
        std::optional<TOAuthRow> row = TryParse(line);
        if (row) {
            Result_.push_back(std::move(*row));
            return true;
        }
        return false;
    }

    void TOAuthParser::Reserve(size_t size) {
        Result_.reserve(Result_.size() + size);
    }

    std::optional<TOAuthRow> TOAuthParser::TryParse(TStringBuf line) {
        NRowParsers::TTskvRow tskv;
        tskv.Reset(line,
                   NRowParsers::TTskvPolicy{
                       .Escape = NRowParsers::ETskvEscapePolicy::Unescape,
                       .Prefix = NRowParsers::ETskvPrefixPolicy::SkipIfFound,
                   });

        Y_ENSURE(tskv.GetField("v") == "1", "Unsupported version");

        TOAuthRow row;

        // https://a.yandex-team.ru/arc_vcs/passport/python/library/historydbloader/hbase_converter.py?rev=73abccfaeee58423158aa11f4bf3e745bdc6f34f#L789
        if (tskv.GetField("action") != "create" ||
            tskv.GetField("target") != "token" ||
            tskv.GetOptionalField("grant_type") != "password" && tskv.GetOptionalField("password_passed") != "1" ||
            tskv.GetOptionalField("has_alias") == "1") {
            return {};
        }

        row.Timestamp = tskv.GetUnixtime("timestamp");
        Y_ENSURE(TryIntFromString<10>(tskv.GetField("uid"), row.Uid), "Invalid UID");

        row.TokenId = tskv.GetField("token_id");
        row.ClientId = tskv.GetField("client_id");
        row.UserIp = tskv.GetField("user_ip");

        row.DeviceName = tskv.GetOptionalField("device_name");
        row.DeviceId = tskv.GetOptionalField("device_id");
        row.Scopes = tskv.GetOptionalField("scopes");

        return row;
    }
}
