#include "yasms_private.h"

#include <passport/infra/daemons/lbchdb/src/processor.h>
#include <passport/infra/daemons/lbchdb/src/crypto/decryptor.h>

#include <passport/infra/libs/cpp/logbroker/row_parsers/utils.h>

namespace NPassport::NLbchdb::NParser {
    static const TYasmsPrivateRow::TFields::TTokens TOKENS = {
        {"error", TYasmsPrivateRow::EKey::Error},
        {"sms", TYasmsPrivateRow::EKey::IsSms},
        {"global_smsid", TYasmsPrivateRow::EKey::GlobalSmsId},
        {"action", TYasmsPrivateRow::EKey::Action},
        {"unixtimef", TYasmsPrivateRow::EKey::Timestamp},
        {"number", TYasmsPrivateRow::EKey::Number},
        {"uid", TYasmsPrivateRow::EKey::Uid},
        {"encryptedtext", TYasmsPrivateRow::EKey::EncryptedText},
        {"tskv_format", TYasmsPrivateRow::EKey::Skip},
        {"unixtime", TYasmsPrivateRow::EKey::Skip},
        {"timestamp", TYasmsPrivateRow::EKey::Skip},
        {"text", TYasmsPrivateRow::EKey::Skip},
    };

    TYasmsPrivateRow::TYasmsPrivateRow()
        : Fields_(TOKENS)
    {
    }

    bool TYasmsPrivateRow::TryParse(TStringBuf line, const NCrypto::TDecryptor& decryptor) {
        Fields_.ResetFromTskv(line);

        if (Fields_.Has(EKey::Error) || Fields_.GetOptional(EKey::IsSms) != "1" || !Fields_.Has(EKey::Number)) {
            return false;
        }

        GlobalSmsId = Fields_.Get(EKey::GlobalSmsId);
        Action = Fields_.Get(EKey::Action);
        Timestamp = NRowParsers::TUtils::ParseUnixtime(Fields_.Get(EKey::Timestamp));
        Number = Fields_.Get(EKey::Number);

        TStringBuf number = Number;
        Y_ENSURE(number.SkipPrefix("+"), "number missing prefix '+': '" << Number << "'");
        if (!TryIntFromString<10>(number, Phone)) {
            TLog::Warning() << "TYasmsPrivateRow: invalid 'number': '" << Number << "'";
            return false;
        }

        TStringBuf uid = Fields_.GetDefault(EKey::Uid, "0");
        Y_ENSURE(TryIntFromString<10>(uid, Uid), "invalid number in 'uid': '" << uid << "'");

        TStringBuf encrypted = Fields_.GetOptional(EKey::EncryptedText);
        Text.clear();
        if (encrypted) {
            try {
                Text = decryptor.Decrypt(encrypted);
            } catch (const std::exception& e) {
                TLog::Warning() << "Failed to decrypt encrypted text in yasms_private: " << e.what();
            }
        }

        return true;
    }

    TYasmsPrivateParser::TYasmsPrivateParser(const NCrypto::TDecryptor& decryptor, NUtils::ILogger* logger)
        : NLb::TParserWithBadLineLog(TLogTypeTraits{}, ELogType::YasmsPrivate, logger)
        , Decryptor_(decryptor)
    {
    }

    TYasmsPrivateParser::TResult TYasmsPrivateParser::Parse(const NLb::TData& data) {
        ParseRows(data);
        return std::move(Result_);
    }

    bool TYasmsPrivateParser::ParseRow(TStringBuf line) {
        TYasmsPrivateRow row;
        if (!row.TryParse(line, Decryptor_)) {
            return false;
        }

        Result_.push_back(std::move(row));
        return true;
    }

    void TYasmsPrivateParser::Reserve(size_t size) {
        Result_.reserve(Result_.size() + size);
    }
}
