#include "processor.h"

#include "sender_unsubsrcibe_lists.h"
#include "extenders/event_extender.h"
#include "extenders/oauth_extender.h"
#include "extenders/sender_delivery_extender.h"
#include "hbase_converters/auth_converter.h"
#include "hbase_converters/event_converter.h"
#include "hbase_converters/mail_user_journal_converter.h"
#include "hbase_converters/oauth_converter.h"
#include "hbase_converters/restore_converter.h"
#include "hbase_converters/yasms_private_converter.h"
#include "helpers/auths_helper.h"
#include "parsers/auth_parser.h"
#include "parsers/event_parser.h"
#include "parsers/mail_user_journal_parser.h"
#include "parsers/oauth_parser.h"
#include "parsers/push_parser.h"
#include "parsers/push_subscription_parser.h"
#include "parsers/restore_parser.h"
#include "parsers/sender_delivery_parser.h"
#include "parsers/yasms_private.h"
#include "yt_converters/auth_converter.h"
#include "yt_converters/mail_user_journal_converter.h"
#include "yt_converters/push_converter.h"
#include "yt_converters/push_subscription_converter.h"
#include "yt_converters/restore_converter.h"
#include "yt_converters/sender_delivery_converter.h"
#include "yt_converters/yasms_private_converter.h"

#include <passport/infra/libs/cpp/utils/log/logger.h>

#include <util/random/easy.h>

namespace NPassport::NLbchdb {
    static const std::map<ELogType, TString> LOG_TYPES = {
        {ELogType::Auth, "auth"},
        {ELogType::BlackboxAuth, "bb_auth"},
        {ELogType::Event, "event"},
        {ELogType::MailUserJournal, "mail_user_journal"},
        {ELogType::OAuth, "oauth"},
        {ELogType::Push, "push"},
        {ELogType::PushSubscription, "push_subscription"},
        {ELogType::Restore, "restore"},
        {ELogType::SenderDelivery, "sender_delivery"},
        {ELogType::YasmsPrivate, "yasms_private"},
    };

    TString TLogTypeTraits::ToString(ELogType type) {
        auto it = LOG_TYPES.find(type);
        Y_VERIFY(it != LOG_TYPES.end());
        return it->second;
    }

    ELogType TLogTypeTraits::FromString(const TString& type) {
        for (const auto& [e, s] : LOG_TYPES) {
            if (type == s) {
                return e;
            }
        }

        Y_FAIL("Unknown log type: %s", type.c_str());
    }

    TProcessor::TProcessor(const NGeobase::IGeobase& geobase,
                           const IIpReg& ipreg,
                           const IUaTraits& uatraits,
                           const TProcessorSettings& settings,
                           NLb::TBadLineLoggers<TLogTypeTraits> badLinesLoggers,
                           NCrypto::TKeyRingPtr decryptorKeyRing,
                           NCrypto::TEncryptor::TKeyRingMap&& encryptorKeyRingMap,
                           std::unique_ptr<TSenderUnsubscribeLists> unsubscribeLists,
                           NDbPool::TDbPool* blackbox,
                           NDbPool::TDbPool* kolmogor)
        : TBase(std::move(badLinesLoggers))
        , Geobase_(geobase)
        , Ipreg_(ipreg)
        , Uatraits_(uatraits)
        , Settings_(settings)
        , Decryptor_(std::move(decryptorKeyRing))
        , Encryptor_(std::move(encryptorKeyRingMap))
        , Blackbox_(blackbox)
        , AuthSampler_(Settings_.AuthSampler, kolmogor)
        , UnsubscribeLists_(std::move(unsubscribeLists))
    {
    }

    TProcessor::~TProcessor() = default;

    void TProcessor::AddUnistat(NUnistat::TBuilder& builder) const {
        Geobase_.AddUnistat(builder);
        Ipreg_.AddUnistat(builder);
        Uatraits_.AddUnistat(builder);

        TBase::AddUnistat(builder);

        AuthSampler_.AddUnistat(builder);
        MailUserJournalModules_.AddUnistat(builder);
        builder.Add(UnistatLastAuthTotal_);
        builder.Add(UnistatLastAuthPassed_);
        builder.Add(UnistatBlackboxErrors_);
    }

    TQueries TProcessor::Proc(const NLb::TDataSet<ELogType>& data) {
        TQueries res;

        for (const NLb::TDataWithType<ELogType>& d : data) {
            res.Merge(ChooseProc(d));
        }

        return res;
    }

    TQueries TProcessor::ChooseProc(const NLb::TDataWithType<ELogType>& data) {
        switch (data.Type) {
            case ELogType::Auth:
            case ELogType::BlackboxAuth:
                return ProcAuth(data.Type, data.Data);
            case ELogType::Event:
                return ProcEvent(data.Data);
            case ELogType::MailUserJournal:
                return ProcMailUserJournal(data.Data);
            case ELogType::OAuth:
                return ProcOAuth(data.Data);
            case ELogType::Push:
                return ProcPush(data.Data);
            case ELogType::PushSubscription:
                return ProcPushSubscription(data.Data);
            case ELogType::SenderDelivery:
                return ProcSenderDelivery(data.Data);
            case ELogType::YasmsPrivate:
                return ProcYasmsPrivate(data.Data);
            case ELogType::Restore:
                return ProcRestore(data.Data);
            case ELogType::COUNT:
                Y_FAIL("Internal error");
        }
    }

    TQueries TProcessor::ProcAuth(ELogType logType, const NLb::TData& data) {
        NParser::TAuthParser parser(logType, &BadLinesLoggers_.GetLogger(logType));
        NParser::TAuthParser::TResult result = parser.Parse(data);
        CountUnistatSignals(logType, parser);

        NExtend::TAuthExtender ext(Geobase_, Ipreg_, Uatraits_);
        NSampler::TBbAuthSampleBulk sampler(AuthSampler_);
        NHBaseConv::TAuthConverter hbaseConv;
        NYtConv::TAuthConverter ytConv(Settings_.TablesLifeTimeConfig);

        sampler.Reserve(result.size());
        hbaseConv.Reserve(result.size());
        ytConv.Reserve(result.size());

        for (NParser::TAuthRow& r : result) {
            NExtend::TAuthExtendedEntry entry = ext.Process(std::move(r));

            hbaseConv.ProcessCommon(entry, Settings_.IsLastauthToHbaseEnabled);
            ytConv.ProcessCommon(entry);

            if (!NHelpers::TAuthsHelper::IsFailedAuthStatus(entry.Row.Status) && NHelpers::TAuthsHelper::IsSampledAuthtype(entry.Row.AuthType)) {
                sampler.Add(
                    NHelpers::TAuthsHelper::BuildSamplingKey(entry, Settings_.AuthSamplingPeriod),
                    std::move(entry));
            } else {
                hbaseConv.ProcessSampled(entry, false);
                ytConv.ProcessSampled(entry, false);
            }
        }

        for (const auto& entry : sampler.Calculate()) {
            hbaseConv.ProcessSampled(entry, true);
            ytConv.ProcessSampled(entry, true);
        }

        return TQueries{
            .Hbase = hbaseConv.Finish(),
            .Yt = ytConv.Finish(UnistatLastAuthTotal_, UnistatLastAuthPassed_),
        };
    }

    TQueries TProcessor::ProcEvent(const NLb::TData& data) {
        NParser::TEventParser parser(Decryptor_, &BadLinesLoggers_.GetLogger(ELogType::Event));
        NExtend::TEventExtender extender(Geobase_, Ipreg_);
        NHBaseConv::TEventConverter conv;

        NParser::TEventParser::TResult parseResult = parser.Parse(data);
        CountUnistatSignals(ELogType::Event, parser);

        for (NParser::TEventRow& r : parseResult) {
            conv.Process(extender.Process(std::move(r)));
        }

        return {.Hbase = conv.Finish()};
    }

    TQueries TProcessor::ProcMailUserJournal(const NLb::TData& data) {
        NParser::TMailUserJournalParser parser(&BadLinesLoggers_.GetLogger(ELogType::MailUserJournal));
        NHBaseConv::TMailUserJournalConverter hbaseConv;
        NYtConv::TMailUserJournalConverter ytConv;

        NParser::TMailUserJournalParser::TResult result = parser.Parse(data);
        CountUnistatSignals(ELogType::MailUserJournal, parser);

        hbaseConv.Reserve(result.size());
        ytConv.Reserve(result.size());

        for (NParser::TMailUserJournalRow& r : result) {
            TStringBuf moduleName = r.GetField("module");
            if (moduleName) {
                MailUserJournalModules_.Add(NUtils::CreateStr("lb.proc.module.", moduleName));
            }

            if (Settings_.IsMailToHbaseEnabled) {
                hbaseConv.Process(r);
            }
            if (Settings_.IsMailToYtEnabled) {
                ytConv.Process(r, Settings_.CompressIfMoreThan, Settings_.TablesLifeTimeConfig);
            }
        }

        return TQueries{
            .Hbase = hbaseConv.Finish(),
            .Yt = ytConv.Finish(),
        };
    }

    TQueries TProcessor::ProcOAuth(const NLb::TData& data) {
        NParser::TOAuthParser parser(&BadLinesLoggers_.GetLogger(ELogType::OAuth));
        NExtend::TOAuthExtender extender(Geobase_, Ipreg_);
        NHBaseConv::TOAuthConverter conv;

        NParser::TOAuthParser::TResult result = parser.Parse(data);
        CountUnistatSignals(ELogType::OAuth, parser);

        for (NParser::TOAuthRow& r : result) {
            conv.Process(extender.Process(std::move(r)));
        }

        return TQueries{.Hbase = conv.Finish()};
    }

    TQueries TProcessor::ProcRestore(const NLb::TData& data) {
        NParser::TRestoreParser parser(Decryptor_, &BadLinesLoggers_.GetLogger(ELogType::Restore));
        NHBaseConv::TRestoreConverter hbaseConv;
        NYtConv::TRestoreConverter ytConv;

        NParser::TRestoreParser::TResult result = parser.Parse(data);
        CountUnistatSignals(ELogType::Restore, parser);

        hbaseConv.Reserve(result.size());
        ytConv.Reserve(result.size());

        for (const NParser::TRestoreRow& p : result) {
            hbaseConv.Process(p);
            ytConv.Process(p, Encryptor_, Settings_.CompressIfMoreThan);
        }

        return TQueries{
            .Hbase = hbaseConv.Finish(),
            .Yt = ytConv.Finish(),
        };
    }

    TQueries TProcessor::ProcSenderDelivery(const NLb::TData& data) {
        Y_VERIFY(Blackbox_);

        NParser::TSenderDeliveryParser parser(
            *UnsubscribeLists_,
            &BadLinesLoggers_.GetLogger(ELogType::SenderDelivery));

        NParser::TSenderDeliveryParser::TResult parseResult = parser.Parse(data);
        CountUnistatSignals(ELogType::SenderDelivery, parser);

        TBlackboxClient bb(*Blackbox_);
        NExtend::TSenderDeliveryExtender extender(bb, UnistatBlackboxErrors_);
        extender.Reserve(parseResult.size());

        for (const NParser::TSenderDeliveryRow& r : parseResult) {
            if (IsSenderDeliveryRowSelected()) {
                extender.Add(r);
            }
        }

        NExtend::TSenderDeliveryExtender::TResult extResult = extender.Finish();

        NYtConv::TSenderDeliveryConverter conv;
        conv.Reserve(extResult.size());
        for (NExtend::TSenderDeliveryExtendedEntry& e : extResult) {
            // NOLINTNEXTLINE(performance-move-const-arg)
            conv.Process(std::move(e));
        }

        return {.Yt = conv.Finish()};
    }

    TQueries TProcessor::ProcYasmsPrivate(const NLb::TData& data) {
        NParser::TYasmsPrivateParser parser(Decryptor_, &BadLinesLoggers_.GetLogger(ELogType::YasmsPrivate));
        NHBaseConv::TYasmsPrivateConverter hbaseConv;
        NYtConv::TYasmsPrivateConverter ytConv(Encryptor_);

        NParser::TYasmsPrivateParser::TResult result = parser.Parse(data);
        CountUnistatSignals(ELogType::YasmsPrivate, parser);

        hbaseConv.Reserve(result.size());
        ytConv.Reserve(result.size());

        for (const NParser::TYasmsPrivateRow& r : result) {
            hbaseConv.Process(r);
            ytConv.Process(r);
        }

        return TQueries{
            .Hbase = hbaseConv.Finish(),
            .Yt = ytConv.Finish(),
        };
    }

    TQueries TProcessor::ProcPush(const NLb::TData& data) {
        NParser::TPushParser parser(&BadLinesLoggers_.GetLogger(ELogType::Push));
        NYtConv::TPushConverter ytConv;

        NParser::TPushParser::TResult result = parser.Parse(data);
        CountUnistatSignals(ELogType::Push, parser);

        ytConv.Reserve(result.size());

        for (NParser::TPushRow& r : result) {
            ytConv.Process(r, Settings_.TablesLifeTimeConfig);
        }

        return TQueries{.Yt = ytConv.Finish()};
    }

    TQueries TProcessor::ProcPushSubscription(const NLb::TData& data) {
        NParser::TPushSubscriptionParser parser(&BadLinesLoggers_.GetLogger(ELogType::PushSubscription));
        NYtConv::TPushSubscriptionConverter ytConv;

        NParser::TPushSubscriptionParser::TResult result = parser.Parse(data);
        CountUnistatSignals(ELogType::PushSubscription, parser);

        ytConv.Reserve(result.size());

        for (const NParser::TPushSubscriptionRow& row : result) {
            ytConv.Process(row, Settings_.PushSubscriptionBucketWidth);
        }

        return TQueries{.Yt = ytConv.Finish()};
    }

    bool TProcessor::IsSenderDeliveryRowSelected() const {
        return static_cast<uint32_t>(Random()) % 100 < Settings_.SenderSampleRatio;
    }
}
