#include "sampler.h"

#include <passport/infra/daemons/lbchdb/src/extenders/auth_extender.h>

#include <passport/infra/libs/cpp/dbpool/handle.h>
#include <passport/infra/libs/cpp/json/writer.h>
#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/log/global.h>

namespace NPassport::NLbchdb::NSampler {
    TInternalSamplerStorage::TInternalSamplerStorage(const TSamplerSettings& settings)
        : Data_(settings.EntryLimit)
    {
    }

    TInternalSamplerStorage::EResult TInternalSamplerStorage::TryAdd(const TString& key) {
        const bool added = AddToData(key);

        return added ? EResult::Added : EResult::Found;
    }

    bool TInternalSamplerStorage::AddToData(const TString& key) {
        std::unique_lock lock(Mutex_);

        if (Data_.Find(key) == Data_.End()) {
            Data_.Insert(
                key,
                0 // dummy
            );
            return true;
        }

        return false;
    }

    TBbAuthSampler::TBbAuthSampler(const TSamplerSettings& settings, NDbPool::TDbPool* kolmogor)
        : Settings_(settings)
        , InternalStorage_(settings)
        , Kolmogor_(kolmogor)
    {
    }

    void TBbAuthSampler::AddUnistat(NUnistat::TBuilder& builder) const {
        builder.Add(UnistatTries_);
        builder.Add(UnistatHit_);
        builder.Add(UnistatKolmogorErrors_);
    }

    void TBbAuthSampler::AddStats(ui64 tries, ui64 hit, ui64 kolmogorErrors) {
        UnistatTries_ += tries;
        UnistatHit_ += hit;
        UnistatKolmogorErrors_ += kolmogorErrors;
    }

    TBbAuthSampleBulk::TBbAuthSampleBulk(TBbAuthSampler& parent)
        : Parent_(parent)
    {
    }

    TBbAuthSampleBulk::~TBbAuthSampleBulk() = default;

    void TBbAuthSampleBulk::Reserve(size_t size) {
        ToSample_.reserve(size);
    }

    void TBbAuthSampleBulk::Add(TString&& key, NExtend::TAuthExtendedEntry&& entry) {
        ++Tries_;

        if (!ToSample_.emplace(std::move(key), std::move(entry)).second) {
            // key was in map already - so it was sampled right now
            ++Hit_;
        }
    }

    TBbAuthSampleBulk::TResult TBbAuthSampleBulk::Calculate() {
        if (ToSample_.empty()) {
            return {};
        }

        ui64 kolmogorErrors = 0;
        rapidjson::Document countersDoc;
        const rapidjson::Value* countersSpace = nullptr;
        try {
            if (Parent_.Kolmogor()) {
                ParseResponse(PerformRequest(CreateKolmoRequest()), countersDoc, countersSpace);
            }
        } catch (const std::exception& e) {
            ++kolmogorErrors;
            TLog::Debug() << "BbAuthSampleBulk: failed to visit kolmogor: " << e.what();
        }

        TResult res;
        res.reserve(ToSample_.size());

        for (auto& [key, entry] : ToSample_) {
            // internal storage should be first:
            // local cache should be usable as fallback if request would fail for another LB chunk
            if (Parent_.InternalStorage().TryAdd(key) == TInternalSamplerStorage::EResult::Found ||
                WasFoundInKolmogor(GetCounterFromResponse(countersSpace, key)))
            {
                ++Hit_;
                continue;
            }

            // new entry
            res.push_back(std::move(entry));
        }

        Parent_.AddStats(Tries_, Hit_, kolmogorErrors);
        return res;
    }

    TString TBbAuthSampleBulk::CreateKolmoRequest() const {
        TString res;
        NJson::TWriter wr(res);
        NJson::TObject root(wr);

        NJson::TArray space(root, Parent_.Settings().KolmogorSpace);
        NJson::TObject obj(space);
        NJson::TArray keys(obj, "keys");

        for (const auto& [key, entry] : ToSample_) {
            keys.Add(key);
        }

        return res;
    }

    static const TString POST = "POST";
    static const TString CONTENT_TYPE = "Content-type";
    static const TString CONTENT_TYPE_JSON = "application/json";

    TString TBbAuthSampleBulk::PerformRequest(TString&& req) const {
        NDbPool::TQuery q("/2/inc");
        q.SetHttpBody(std::move(req));
        q.SetHttpMethod(POST);
        q.AddHttpHeader(CONTENT_TYPE, CONTENT_TYPE_JSON);

        // Do not retry this request: there could be timeout.
        // In this case request could be actually performed, but client won't get response.
        // After retry counter will have value >= 2 - and it will be useless:
        //   "value == 1" means "new key"

        NDbPool::TBlockingHandle h(*Parent_.Kolmogor());
        NDbPool::THttpResponse response = h.Query(std::move(q))->ToHttpResponse();
        Y_ENSURE(response.Status == 200,
                 "request failed: " << response.Status << ": " << response.Body);

        return std::move(response.Body);
    }

    void TBbAuthSampleBulk::ParseResponse(const TString& resp,
                                          rapidjson::Document& doc,
                                          const rapidjson::Value*& space) const {
        Y_ENSURE(NJson::TReader::DocumentAsObject(resp, doc),
                 "invalid json from kolmogor: " << resp);
        Y_ENSURE(NJson::TReader::MemberAsObject(doc, Parent_.Settings().KolmogorSpace.c_str(), space),
                 "missing space '" << Parent_.Settings().KolmogorSpace << "' in response: " << resp);
    }

    std::optional<ui64> TBbAuthSampleBulk::GetCounterFromResponse(const rapidjson::Value* space, const TString& key) {
        if (!space) {
            return {};
        }

        ui64 res;
        const rapidjson::Value* obj = nullptr;
        if (!(NJson::TReader::MemberAsObject(*space, key.c_str(), obj) &&
              NJson::TReader::MemberAsUInt64(*obj, "value", res))) {
            TLog::Warning() << "BbAuthSampleBulk: key was requested in kolmogor, but missing in response: "
                            << key;
            return {};
        }

        return res;
    }

    bool TBbAuthSampleBulk::WasFoundInKolmogor(std::optional<ui64> val) {
        return val && val > 1;
    }
}
