#include "auths.h"

#include <passport/infra/daemons/lbchdb/src/yt_converters/utils.h>

#include <passport/infra/daemons/lbchdb/src/helpers/auths_helper.h>

#include <passport/infra/libs/cpp/yt/yt_impl.h>

#include <yt/yt/client/table_client/comparator.h>
#include <yt/yt/client/table_client/helpers.h>
#include <yt/yt/client/table_client/logical_type.h>
#include <yt/yt/client/table_client/name_table.h>

namespace NPassport::NLbchdb::NYtConv::NAuth {
    static NYT::NYson::TYsonString BuildData(const NExtend::TAuthExtendedEntry& entry) {
        TStringStream stream;
        stream.Reserve(1024);

        NYT::NYson::TBufferedBinaryYsonWriter writer(&stream);
        writer.OnBeginMap();

        auto writeOptional = [&writer](const TStringBuf key, const TStringBuf value) {
            if (value.empty()) {
                return;
            }

            writer.OnKeyedItem(key);
            writer.OnStringScalar(value);
        };

        writeOptional("host_id", ToString(entry.Row.HostId));
        writeOptional("login", entry.Row.Login);
        writeOptional("sid", entry.Row.Sid);
        writeOptional("yandexuid", entry.Row.YandexUid);
        writeOptional("comment", entry.Row.Comment);

        writeOptional("user_ip", NExtend::TIpData::ChooseUserIp(entry.Row));
        if (entry.IpData) {
            const NExtend::TIpData& data = *entry.IpData;
            if (data.GeoId) {
                writeOptional("ip.geoid", ToString(*data.GeoId));
            }
            if (data.As) {
                writeOptional("ip.as_list", *data.As);
            }
            if (data.IsYandex) {
                writeOptional("ip.is_yandex", "1");
            }
        }

        if (entry.UaData) {
            const NExtend::TUserAgentData& data = *entry.UaData;

            writeOptional("browser.name", data.BrowserName);
            writeOptional("browser.version", data.BrowserVersion);

            writeOptional("os.name", data.OsName);
            writeOptional("os.family", data.OsFamily);
            writeOptional("os.version", data.OsVersion);
        }

        writer.OnEndMap();
        writer.Flush();

        return NYT::NYson::TYsonString(stream.Str());
    }

    TAuthEntry TAuthEntry::Create(const NExtend::TAuthExtendedEntry& entry, bool isSamplingEnabled) {
        return TAuthEntry{
            .Uid = entry.Row.Uid,
            .ReversedTimestamp = TUtils::ReverseTimeT(entry.Row.Timestamp.MicroSeconds()),
            .KeyUniquePart = NHelpers::TAuthsHelper::BuildKeyUniquePart(entry, isSamplingEnabled),
            .Type = TString(entry.Row.AuthType),
            .Status = TString(entry.Row.Status),
            .ClientName = TString(entry.Row.ClientName),
            .Data = BuildData(entry),
        };
    }

    NYt::TWriteQuery TAuthsQueryConverter::Convert(const TString& table, size_t offset, size_t count) const {
        return ConvertImpl(table, GetSpan(offset, count));
    }

    static NYT::NTableClient::TNameTablePtr CreateNameTable() {
        NYT::NTableClient::TNameTablePtr res = NYT::New<NYT::NTableClient::TNameTable>();
        res->RegisterName("uid");
        res->RegisterName("reversed_timestamp");
        res->RegisterName("unique_part");
        res->RegisterName("type");
        res->RegisterName("status");
        res->RegisterName("client_name");
        res->RegisterName("data");
        return res;
    }
    static const NYT::NTableClient::TNameTablePtr NAME_TABLE = CreateNameTable();

    NYt::TWriteQuery TAuthsQueryConverter::ConvertImpl(const TString& table,
                                                       std::span<const TAuthEntry> data) {
        NYt::TWriteQuery res;
        res.Impl = std::make_unique<NYt::TWriteQueryImpl>();

        NYT::NTableClient::TUnversionedRowsBuilder builder;
        builder.ReserveRows(data.size());

        for (const auto& entry : data) {
            builder.AddRow(
                entry.Uid,
                entry.ReversedTimestamp,
                entry.KeyUniquePart,
                entry.Type,
                entry.Status,
                entry.ClientName,
                entry.Data);
        }

        res.Impl->Subqueries.emplace_back(NYt::TWriteSubQuery{
            .Path = table,
            .NameTable = NAME_TABLE,
            .Range = builder.Build(),
        });

        return res;
    }

    const TString TAuths::AUTHS_TABLE_DIR = "auths";
    const TString TAuths::FAILED_AUTHS_TABLE_NAME = "failed_auths/failed_auths";

    NYt::TTableSchema TAuths::CreateYtSchema() {
        NYt::TTableSchema res;
        res.Impl = std::make_shared<NYt::TTableSchemaImpl>();

        res.Impl->Schema = NYT::NTableClient::TTableSchema(
            std::vector<NYT::NTableClient::TColumnSchema>({
                NYT::NTableClient::TColumnSchema(
                    "uid",
                    SimpleLogicalType(NYT::NTableClient::ESimpleLogicalValueType::Uint64),
                    NYT::NTableClient::ESortOrder::Ascending),
                NYT::NTableClient::TColumnSchema(
                    "reversed_timestamp",
                    SimpleLogicalType(NYT::NTableClient::ESimpleLogicalValueType::Int64),
                    NYT::NTableClient::ESortOrder::Ascending),
                NYT::NTableClient::TColumnSchema(
                    "unique_part",
                    SimpleLogicalType(NYT::NTableClient::ESimpleLogicalValueType::String),
                    NYT::NTableClient::ESortOrder::Ascending),
                NYT::NTableClient::TColumnSchema("type", NYT::NTableClient::ESimpleLogicalValueType::String),
                NYT::NTableClient::TColumnSchema("status", NYT::NTableClient::ESimpleLogicalValueType::String),
                NYT::NTableClient::TColumnSchema("client_name", NYT::NTableClient::ESimpleLogicalValueType::String),
                NYT::NTableClient::TColumnSchema("data", NYT::NTableClient::ESimpleLogicalValueType::Any),
            }),
            true,
            true);

        return res;
    }
}

using namespace NPassport::NLbchdb::NYtConv::NAuth;

template <>
void Out<TAuthEntry>(IOutputStream& o, const TAuthEntry&) {
    o << "TAuthEntry" << Endl;
}
