#include "last_auth.h"

#include <passport/infra/daemons/lbchdb/src/hbase_converters/auth/last_auth.h>
#include <passport/infra/daemons/lbchdb/src/helpers/auths_helper.h>

#include <passport/infra/libs/cpp/yt/yt_impl.h>

#include <yt/yt/client/table_client/helpers.h>
#include <yt/yt/client/table_client/name_table.h>
#include <yt/yt/client/table_client/row_buffer.h>

namespace NPassport::NLbchdb::NYtConv::NAuth {
    const TString TLastAuth::TABLE_NAME = "lastauth/lastauth";

    std::optional<TLastAuthEntry> TLastAuthEntry::Create(const NExtend::TAuthExtendedEntry& entry) {
        if (!NHelpers::TAuthsHelper::IsSuccessAuth(entry.Row, true)) {
            return {};
        }

        return TLastAuthEntry{
            .Uid = entry.Row.Uid,
            .AuthType = TString(entry.Row.AuthType),
            .Timestamp = entry.Row.Timestamp,
        };
    }

    NYt::TWriteQuery TLastAuthConverter::Convert(const TString& table,
                                                 size_t offset,
                                                 size_t count) const {
        return ConvertImpl(table, GetSpan(offset, count));
    }

    static NYT::NTableClient::TNameTablePtr CreateNameTable() {
        NYT::NTableClient::TNameTablePtr res = NYT::New<NYT::NTableClient::TNameTable>();
        res->RegisterName("uid");
        res->RegisterName("authtype");
        res->RegisterName("timestamp");
        return res;
    }
    static const NYT::NTableClient::TNameTablePtr NAME_TABLE = CreateNameTable();

    NYt::TWriteQuery TLastAuthConverter::ConvertImpl(
        const TString& table,
        std::span<const TLastAuthEntry> data) {
        NYt::TWriteQuery res;

        res.Impl = std::make_unique<NYt::TWriteQueryImpl>();
        NYt::TWriteSubQuery& sub = res.Impl->Subqueries.emplace_back();

        sub.Path = table;
        sub.NameTable = NAME_TABLE;

        sub.RowBuffer = NYT::New<NYT::NTableClient::TRowBuffer>();
        std::vector<NYT::NTableClient::TUnversionedRow> vecRows;
        vecRows.reserve(data.size());

        for (const TLastAuthEntry& d : data) {
            int colIdx = 0;
            NYT::NTableClient::TUnversionedRowBuilder builder;

            auto addUint = [&builder, &colIdx](ui64 val, bool aggregate) {
                using namespace NYT::NTableClient;
                auto flags = aggregate ? EValueFlags::Aggregate : EValueFlags::None;
                builder.AddValue(MakeUint64Value<TUnversionedValue>(val, colIdx++, flags));
            };
            auto addString = [&builder, &colIdx](const TStringBuf val) {
                using namespace NYT::NTableClient;
                builder.AddValue(MakeStringValue<TUnversionedValue>(val, colIdx++, EValueFlags::None));
            };

            addUint(d.Uid, false);
            addString(d.AuthType);
            addUint(d.Timestamp.MicroSeconds(), true);

            vecRows.push_back(sub.RowBuffer->CaptureRow(builder.GetRow()));
        }

        res.Size = sub.RowBuffer->GetSize();
        sub.Range = MakeSharedRange(std::move(vecRows), sub.RowBuffer);

        return res;
    }

    void TLastAuthFilter::Reserve(size_t size) {
        Y_VERIFY(!IsFinished_, "Internal error: Filter finished operation");
        Entries_.reserve(size);
    }

    void TLastAuthFilter::Add(TLastAuthEntry&& e) {
        Y_VERIFY(!IsFinished_, "Internal error: Filter finished operation");

        const TInstant timestamp = e.Timestamp;

        TString key = NUtils::CreateStr(e.Uid, "|", e.AuthType);

        TLastAuthEntry& res = Entries_.emplace(std::move(key), std::move(e)).first->second;
        res.Timestamp = std::max(res.Timestamp, timestamp);
    }

    TLastAuthFilter::TEntries TLastAuthFilter::ExtractEntries() {
        Y_VERIFY(!IsFinished_, "Internal error: Filter finished operation");
        IsFinished_ = true;

        return std::move(Entries_);
    }
}

using namespace NPassport::NLbchdb::NYtConv::NAuth;

template <>
void Out<TLastAuthEntry>(IOutputStream& o, const TLastAuthEntry& value) {
    o << "uid=" << value.Uid
      << ". authtype=" << value.AuthType
      << ". ts=" << value.Timestamp
      << Endl;
}
