#include "auth_converter.h"

#include <passport/infra/daemons/lbchdb/src/helpers/auths_helper.h>

namespace NPassport::NLbchdb::NYtConv {
    TAuthConverter::TAuthConverter(const TTablesLifeTimeConfig& ttlConfig)
        : TtlConfig_(ttlConfig)
        , FailedAuthsConverter_(std::make_unique<NAuth::TAuthsQueryConverter>())
    {
    }

    TAuthConverter::~TAuthConverter() = default;

    void TAuthConverter::Reserve(size_t size) {
        Reserve_ = size;
        LastAuthFilter_.Reserve(size);
        FailedAuthsConverter_->Reserve(size);
    }

    void TAuthConverter::ProcessCommon(const NExtend::TAuthExtendedEntry& entry) {
        Y_VERIFY(!IsFinished_, "Internal error: Converter finished operation");

        if (std::optional e = NAuth::TLastAuthEntry::Create(entry); e) {
            ++LastAuthTotal_;
            LastAuthFilter_.Add(std::move(*e));
        }
    }

    void TAuthConverter::ProcessSampled(const NExtend::TAuthExtendedEntry& entry, bool isSamplingEnabled) {
        Y_VERIFY(!IsFinished_, "Internal error: Converter finished operation");

        if (NHelpers::TAuthsHelper::IsFailedAuthStatus(entry.Row.Status)) {
            FailedAuthsConverter_->Add(NAuth::TAuthEntry::Create(entry, isSamplingEnabled));
        } else {
            if (!TtlConfig_.IsDateActual(entry.Row.Timestamp, TtlConfig_.AuthsTtl)) {
                TLog::Warning() << "AuthConverter: got non-actual date in event: " << entry.Row.Timestamp
                                << ". uid=" << entry.Row.Uid;
                return;
            }

            NAuth::TAuthsQueryConverterPtr& converter = AuthsConverters_[NYt::TMonthlyTable(entry.Row.Timestamp)];
            if (!converter) {
                converter = std::make_unique<NAuth::TAuthsQueryConverter>();
                converter->Reserve(Reserve_);
            }

            converter->Add(NAuth::TAuthEntry::Create(entry, isSamplingEnabled));
        }
    }

    NYt::TWriteQueries<TString> TAuthConverter::Finish(
        NUnistat::TSignalDiff<>& total,
        NUnistat::TSignalDiff<>& passed) {
        Y_VERIFY(!IsFinished_, "Internal error: Converter finished operation");
        IsFinished_ = true;

        NAuth::TLastAuthConverterPtr lastAuthConverter = FilterLastAuth();
        total += LastAuthTotal_;
        passed += lastAuthConverter->size();

        NYt::TWriteQueries<TString> res;
        res.ByTable.emplace(NAuth::TLastAuth::TABLE_NAME, std::move(lastAuthConverter));
        res.ByTable.emplace(NAuth::TAuths::FAILED_AUTHS_TABLE_NAME, std::move(FailedAuthsConverter_));
        for (auto& [key, value] : AuthsConverters_) {
            res.ByTable.emplace(
                NUtils::CreateStr(NAuth::TAuths::AUTHS_TABLE_DIR, "/", key.TableName()),
                std::move(value));
        }

        return res;
    }

    NAuth::TLastAuthConverterPtr TAuthConverter::FilterLastAuth() {
        NAuth::TLastAuthFilter::TEntries entries = LastAuthFilter_.ExtractEntries();

        std::unique_ptr res = std::make_unique<NAuth::TLastAuthConverter>();
        res->Reserve(entries.size());

        for (auto& [key, entry] : entries) {
            res->Add(std::move(entry));
        }

        return res;
    }
}
