#include "common.h"

#include <library/cpp/streams/brotli/brotli.h>
#include <library/cpp/streams/zstd/zstd.h>

#include <yt/yt/core/yson/writer.h>

#include <util/stream/str.h>
#include <util/stream/zlib.h>

namespace NPassport::NLbchdb::NYtConv {
    TString TCompressor::Compress(const TStringBuf blob,
                                  ECompressionCodec codec,
                                  std::optional<size_t> level) {
        TStringStream res;
        res.Reserve(blob.size() / 10);

        std::unique_ptr<IOutputStream> coder;

        switch (codec) {
            case ECompressionCodec::GZip:
                coder = std::make_unique<TZLibCompress>(&res, ZLib::GZip, level.has_value() ? *level : 6);
                break;
            case ECompressionCodec::Brotli:
                coder = std::make_unique<TBrotliCompress>(&res, level.has_value() ? *level : 8);
                break;
            case ECompressionCodec::ZStd:
                coder = std::make_unique<TZstdCompress>(&res, level.has_value() ? *level : 3);
                break;
        }

        coder->Write(blob.Data(), blob.Size());
        coder->Finish();

        return std::move(res.Str());
    }

    static const THashMap<TCompressor::ECompressionCodec, TString> ENCODINGS_NAMES = {
        {TCompressor::ECompressionCodec::GZip, "gzip"},
        {TCompressor::ECompressionCodec::Brotli, "brotli"},
        {TCompressor::ECompressionCodec::ZStd, "zstd"},
    };

    void TYsonSerializer::SerializeEncrypted(NYT::NYson::TBufferedBinaryYsonWriter& writer,
                                             const TStringBuf blob,
                                             const NCrypto::TEncryptor& encryptor,
                                             NCrypto::TEncryptor::EKeyRing ring,
                                             const TEncryptionSettings& settings) {
        Y_ENSURE(ENCODINGS_NAMES.contains(settings.Codec), "Unknown compression codec: " << int(settings.Codec));

        writer.OnBeginMap();

        writer.OnKeyedItem("v");
        writer.OnUint64Scalar(1);

        TStringBuf toEncrypt = blob;

        TString compressed;
        if (blob.size() > settings.CompressIfMoreThan) {
            writer.OnKeyedItem("codec");
            writer.OnStringScalar(ENCODINGS_NAMES.at(settings.Codec));

            writer.OnKeyedItem("size");
            writer.OnUint64Scalar(blob.size());

            compressed = TCompressor::Compress(blob, settings.Codec, settings.Level);
            toEncrypt = compressed;
        }

        NCrypto::TEncryptor::TEncryptResult encrypted = encryptor.Encrypt(toEncrypt, ring);

        writer.OnKeyedItem("keyid");
        writer.OnUint64Scalar(encrypted.KeyId);

        writer.OnKeyedItem("iv");
        writer.OnStringScalar(encrypted.Crypt.Iv);
        writer.OnKeyedItem("text");
        writer.OnStringScalar(encrypted.Crypt.Text);
        writer.OnKeyedItem("tag");
        writer.OnStringScalar(encrypted.Crypt.Tag);

        writer.OnEndMap();
    }
}
